//==========================================================================
// maswitch.c
//
// Manages the "Parameter Edit" (IDD_PARAMEDIT) dialog for audio controls that
// are of the switch (MIXERCONTROL_CT_CLASS_SWITCH) class. This displays the
// settings of the selected audio control for the selected audio line of the
// currently open Mixer device.
//
// This dialog is opened when the user clicks the "Value" button for a switch
// type of audio control in the Main Window's listbox full of audio control
// names.
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
// PURPOSE.
//
// Copyright (C) 1993 - 1997  Microsoft Corporation.  All Rights Reserved.
// Modified 2001 by Jeff Glatt
//==========================================================================

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>

#include "resource.h"
#include "mixapp.h"





//
typedef struct tMACONTROLINSTANCE_SWITCH
{
    MIXERCONTROLDETAILS_BOOLEAN     pmxcd_f[];
} MACONTROLINSTANCE_SWITCH, *PMACONTROLINSTANCE_SWITCH;





// ************************* changeSwitchLine() *************************
// Called when our Main Window is notified that a line in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_LINE_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD dwLineID:		ID of the line whose state has changed.
//
// RETURNS:
//		1 if the line is active, 0 if inactive.

BOOL changeSwitchLine(HWND hwnd, HMIXER hmx, DWORD dwLineID)
{
	HWND			htxt;
	MMRESULT		err;
	BOOL			fActive;

	// Is the line that was changed the same as the line to which our currently
	// displayed parameter belongs?
//	if (CurrMixerLine.dwLineID != dwLineID)
//	{
		// Uh oh. The Mixer is informing us of a change to another line (ie, not
		// the one that the user has currently selected in the main window).

		// Just update for the currently selected line
//		dwLineID = CurrMixerLine.dwLineID;
//	}

	// Get info on the line that changed
	CurrMixerLine.cbStruct = sizeof(MIXERLINE);

	if ((err = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &CurrMixerLine, MIXER_GETLINEINFOF_LINEID)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on line ID #%u!", err, CurrMixerLine.dwLineID);
		return(FALSE);
	}

	// Get the current state of the line that changed -- active or inactive
	fActive       = (0 != (MIXERLINE_LINEF_ACTIVE & CurrMixerLine.fdwLine));

	// Refresh the "Line Info"
	htxt = GetDlgItem(hwnd, IDD_MACONTROL_TXT_LINEINFO);
	doWindowText(htxt, "(%s), '%s', %s, %s",
					(0 != (MIXERLINE_LINEF_SOURCE & CurrMixerLine.fdwLine)) ? (LPSTR)"src" : (LPSTR)"DST",
					(LPSTR)CurrMixerLine.szShortName,
					fActive ? (LPSTR)"ACTIVE" : (LPSTR)"inactive",
					(0 != (MIXERLINE_LINEF_DISCONNECTED & CurrMixerLine.fdwLine)) ? (LPSTR)"DISCONNECTED" : (LPSTR)"connected");

	return(fActive);
}




// ************************ changeSwitchParameter() ************************
// Called when our Main Window is notified that a parameter in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_CONTROL_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD controlID:	ID of the parameter whose state has changed.
//
// RETURNS:
//		0 if success, or non-zero if an error # from the Mixer API.

MMRESULT changeSwitchParameter(HWND hwnd, HMIXER hmx, DWORD dwControlID)
{
	MMRESULT						err;
	MACONTROLINSTANCE_SWITCH		*pmaci_switch;
	MIXERCONTROLDETAILS_BOOLEAN		*pmxcd_f;
	BOOL							fValue;
	UINT							cChannels, cMultipleItems;
	UINT							uIndex;
	MIXERCONTROLDETAILS				mixerControlDetails;
	HWND							hwndFocus;
	HWND							hwndChild;

	// Get the switch with the focus
	if (!(hwndFocus = GetFocus()))
	{
		// If no switch has the focus, assume the first of the individual switches
		hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
	}
	else
	{
		// Get the ID of the list item with the focus
		uIndex = GetDlgCtrlID(hwndFocus);

		// If not one of the switches, then assume the first channel's first switch. It may
		// be that this was called as a result of the Mixer device informing us
		// of a change to our parameter, not because the user selected some switch
		if (uIndex < IDD_MACONTROL_MULTICHANNEL_BASE)
		{
			hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
		}
	}

	// Retrieve the array of MIXERCONTROLDETAILS_BOOLEAN structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_switch = (MACONTROLINSTANCE_SWITCH *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_f = &pmaci_switch->pmxcd_f[0];

	// ================== Update the individual switches =================

	cChannels = (UINT)CurrMixerLine.cChannels;
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

    // Get the current values of each individual list item of each channels. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;							// This is the # of channels
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;	// The # of items per channel
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_BOOLEAN);
	mixerControlDetails.paDetails = pmxcd_f;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}
	
	cMultipleItems = 1;
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;


	// Determine the total # of individual faders for all items
	cChannels *= cMultipleItems;

	// Start with the first list item
	uIndex = 0;

	while (cChannels--)
	{
		fValue = (BOOL)pmxcd_f[uIndex].fValue;

		hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE + uIndex);

		// If this is the currently selected list item, print out the value
		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			doWindowText(hwndChild, "fValue=%lu", fValue);
		}

		// If it's state is not already set as desired, set its state
		if (fValue != (BOOL)IsDlgButtonChecked(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE + uIndex))
		{
			CheckDlgButton(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE + uIndex, fValue);
		}

		// Next scrollbar window ID
		uIndex++;
    }

	// ================== Update the uniform (ie, master) switches =================

	// Get info upon each item's master switch value. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = 1;										// Just get the master fader values -- not all individual ones
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;		// This is how many master faders
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_BOOLEAN);
	mixerControlDetails.paDetails = pmxcd_f;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	while (cMultipleItems--)
	{
		fValue = (BOOL)pmxcd_f[cMultipleItems].fValue;

		hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems);

		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			doWindowText(hwndChild, "fValue=%lu", fValue);
		}

		// If it's state is not already set as desired, set its state
		if (fValue != (BOOL)IsDlgButtonChecked(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems))
		{
			CheckDlgButton(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems, fValue);
		}
	}

	// Success
	return(0);
}





// *************************** setSwitchParameter() ***************************
// Called when a graphical switch is selected by the user to set the value
// of some audio parameter of switch type.
//
// ARGS:
//		HWND hwnd:		Handle to Parameter Edit dialog for switch type of
//						parameter.
//
//      HWND hcheck:	Handle to switch graphical control.
//
// RETURNS:
//		0 if success, or non-zero if an error # from the Mixer API.

MMRESULT setSwitchParameter(HWND hwnd, HWND hcheck)
{
	MACONTROLINSTANCE_SWITCH		*pmaci_switch;
	MIXERCONTROLDETAILS_BOOLEAN		*pmxcd_f;
	MMRESULT						err;
	UINT							cChannels;
	UINT							uIndex;
	MIXERCONTROLDETAILS				mixerControlDetails;

	// Retrieve the array of MIXERCONTROLDETAILS_BOOLEAN structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_switch = (MACONTROLINSTANCE_SWITCH *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_f = &pmaci_switch->pmxcd_f[0];

	uIndex = GetDlgCtrlID(hcheck);

	if (uIndex < IDD_MACONTROL_UNIFORM_BASE)
	{
		cChannels = (UINT)CurrMixerLine.cChannels;
		if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
			cChannels = 1;

		uIndex -= IDD_MACONTROL_MULTICHANNEL_BASE;
	}
	else
	{
		cChannels = 1;

		uIndex -= IDD_MACONTROL_UNIFORM_BASE;
	}

	// Get the current value of the switch
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_BOOLEAN);
	mixerControlDetails.paDetails = pmxcd_f;

	if ((err = mixerGetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	// Get current value, toggle, and store the new value for this switch
	pmxcd_f[uIndex].fValue = (0 == IsDlgButtonChecked(hwnd, uIndex));

	// Send this new value to the Mixer
	if ((err = mixerSetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_SETCONTROLDETAILSF_VALUE)))
    {
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u setting value of parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	// Set the graphical switch's state
    CheckDlgButton(hwnd, GetDlgCtrlID(hcheck), (BOOL)pmxcd_f[uIndex].fValue);

	// Success
    return(0);
}





// ************************** initSwitchDlg() **************************
// Called by switchDlgProc()'s WM_INITDIALOG to initialize the "Parameter
// Edit" (IDD_PARAMEDIT) dialog prior to displaying it for the first
// time.
//
// NOTE: The global 'CurrParameter' MIXERCONTROL struct must already be
// filled in with the current parameter, and 'CurrMixerLine' MIXERLINE
// struct must be filled in with its line.
//
// ARGS:
//		HWND hwnd:					Handle to "Parameter Edit" dialog.
//
//		LPMACONTROLINSTANCE pmaci:	Pointer to MACONTROLINSTANCE struct
//									filled in with the values of the
//									parameter being displayed.
//
// RETURNS:
//		0 if success, or non-zero if an error (-4 if not a type of switch
//		understood by this app, -1 if memory error, -5 if a window
//		can't be created, or other error # from the Mixer API).

MMRESULT initSwitchDlg(HWND hwnd)
{
#define FCB_DEF_STYLE   (WS_VISIBLE | WS_CHILD | BS_CHECKBOX | WS_TABSTOP)

	static const TCHAR			szButton[]  = TEXT("button");

	PMACONTROLINSTANCE_SWITCH	pmaci_switch;
	UINT						u, v;
	HWND						hwndChild;
	RECT                        rc;
	int                         cycap;
	UINT                        cChannels, cMultipleItems;
	UINT                        uIndex;

	// Check that it's a legitimate parameter type for the switch class
	switch (CurrParameter.dwControlType)
	{
		case MIXERCONTROL_CONTROLTYPE_BOOLEAN:
		case MIXERCONTROL_CONTROLTYPE_ONOFF:
		case MIXERCONTROL_CONTROLTYPE_MUTE:
		case MIXERCONTROL_CONTROLTYPE_MONO:
		case MIXERCONTROL_CONTROLTYPE_LOUDNESS:
		case MIXERCONTROL_CONTROLTYPE_STEREOENH:
			break;

		default:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Invalid switch type: 0x%.08X!", CurrParameter.dwControlType);
			return((MMRESULT)-4);
		}
	}

	// Set the dialog's title to the name of the parameter being edited
	SetWindowText(hwnd, &CurrParameter.szName[0]);

	// Display the minimum and maximum limits
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_BOUNDS);
	doWindowText(hwndChild, "dwMinimum=%lu, dwMaximum=%lu", CurrParameter.Bounds.dwMinimum, CurrParameter.Bounds.dwMaximum);

	// The steps are not applicable for a switch class, so hide that display
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_METRICS);
	ShowWindow(hwndChild, SW_HIDE);

	// Determine how many arrays we'll need, based upon how many items there are,
	// and how many channels are in each item

	// Assume not mono
	cChannels = (UINT)CurrMixerLine.cChannels;

	// If a uniform parameter, then we have only a mono adjustment
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

	// Assume only 1 item
	cMultipleItems = 1;

	// If multiple items, then get how many items
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Get a MACONTROLINSTANCE_SWITCH struct with an array of MIXERCONTROLDETAILS_BOOLEAN
	// structs for this parameter. The number of array structs are cMultipleItems * cChannels
	// so that we can set/retrieve all individual items on all channels simultaneously
	v = sizeof(MACONTROLINSTANCE_SWITCH) + (cChannels * cMultipleItems * sizeof(MIXERCONTROLDETAILS_BOOLEAN));
	if (!(pmaci_switch = (MACONTROLINSTANCE_SWITCH *)LocalAlloc(LPTR, v)))
 	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't get memory for switches!");
 		return((MMRESULT)-1);
 	}

	// Save the pointer in the dialog's DWL_USER field so that other functions can retrieve it
	SetWindowLong(hwnd, DWL_USER, (LPARAM)pmaci_switch);


	// Figure out how wide to make each switch based upon how many items we need
	// to squeeze into the IDC_PARAMEDIT_MULTICHANNEL groupbox
	cycap = GetSystemMetrics(SM_CYCAPTION);

	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_MULTICHANNEL);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.bottom = rc.top + cycap;

	// Create the switches

	// ================== Create the individual item switches =================

	// Start with a unique window ID
	uIndex = IDD_MACONTROL_MULTICHANNEL_BASE;

	// For each channel...
	for (u = 0; u < cChannels; u++)
	{
		// and each item on this channel...
		for (v = 0; v < cMultipleItems; v++)
		{
			// Create a switch, and then increment the window ID #
			if (!(hwndChild = CreateWindow(szButton, CurrParameter.szShortName, FCB_DEF_STYLE,
								rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
								hwnd, (HMENU)(uIndex++),
								MyInstance, NULL)))
			{
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create switch!");
				return((MMRESULT)-5);
			}

			// Move over for next item's switch
			rc.top += cycap + 4;
			rc.bottom += cycap + 4;
		}

		// Add separation between this switch and the next channel's switch
		rc.top += cycap;
		rc.bottom += cycap;
	}

	// ====================== Update the master switches =====================

	// Figure out how wide to make each switch based upon how many switches
	// we have to squeeze into the IDC_PARAMEDIT_UNIFORM groupbox
	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_UNIFORM);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.bottom = rc.top + cycap;

	for (v = 0; v < cMultipleItems; v++)
	{
		if (!(hwndChild = CreateWindow(szButton, CurrParameter.szShortName, FCB_DEF_STYLE,
                              rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
                              hwnd, (HMENU)(IDD_MACONTROL_UNIFORM_BASE + v),
                              MyInstance, NULL)))
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create switch!");
			return((MMRESULT)-5);
		}

		rc.top += cycap + 4;
		rc.bottom += cycap + 4;
	}

	// =========================================================================

	// Send MM_MIXM_LINE_CHANGE and MM_MIXM_CONTROL_CHANGE messages to
	// IDD_PARAMEDIT dialog so that it updates its window per the
	// current settings of the current line and audio parameter. (ie,
	// Just like its window procedure would do if the operating system
	// sent it these messages)
	SendMessage(hwnd, MM_MIXM_LINE_CHANGE, (WPARAM)MixerHandle, CurrMixerLine.dwLineID);
	SendMessage(hwnd, MM_MIXM_CONTROL_CHANGE, (WPARAM)MixerHandle, CurrParameter.dwControlID);

	// Success
	return(0);
}





// *************************** switchDlgProc() ****************************
// Dialog procedure for the "Parameter Edit" (IDD_PARAMEDIT) dialog when
// it is displaying the settings of a switch (MIXERCONTROL_CT_CLASS_SWITCH)
// type of audio parameter.
//
// This dialog is opened when the user clicks the "Value" button for a
// switch type of control in the main window's dialog's listbox full of
// parameter names.
//
//	ARGS:
//		HWND hwnd:		Handle to DLG_MIXAPP_CONTROL dialog.
//
//		UINT uMsg:		The message type.
//
//		WPARAM wParam:	Depends upon uMsg.
//
//		LPARAM lParam:	Depends upon uMsg.
//
//	RETURNS:
//		Typically, TRUE if the message is handled, or FALSE if not.

BOOL APIENTRY switchDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			MMRESULT	err;

			// Create some child control windows, such as faders or checkmarks, which
			// will display the values of the channel masters and individual channels
			// for this audio parameter
			if ((err = initSwitchDlg(hwnd)))

			// If an error, close the dialog
			{
				EndDialog(hwnd, err);
			}

			// If success, save the handle of this dialog in a global so that we'll
			// know it's open in case the operating system tells us that this audio
			// parameter has been updated
			else
				ghdlgControl = hwnd;

			return(TRUE);
		}

		// The operating system is notifying us that some line of the mixer we
		// currently have open is changed. We may need to now fetch that line's
		// new settings and update our window (assuming that we're currently
		// displaying that line's settings)
		case MM_MIXM_LINE_CHANGE:
		{
			changeSwitchLine(hwnd, (HMIXER)wParam, lParam);
			return(TRUE);
		}

		// The operating system is notifying us that some audio parameter of the
		// mixer we currently have open is changed. We may need to now fetch
		// that parameter's new settings and update our window (assuming that
		// we're currently displaying that parameter's settings)
		case MM_MIXM_CONTROL_CHANGE:
		{
			changeSwitchParameter(hwnd, (HMIXER)wParam, lParam);
			return(TRUE);
		}

		case WM_COMMAND:
		{
			uMsg = GET_WM_COMMAND_ID(wParam, lParam);
			if (IDC_PARAMEDIT_MULTICHANNEL <= uMsg)
			{
				setSwitchParameter(hwnd, GET_WM_COMMAND_HWND(wParam, lParam));
				return(TRUE);
			}

			// If OK button, or window close button, close the dialog
			// and free memory
			switch (uMsg)
			{
				case IDCANCEL:
				{
					HLOCAL		hl;

					// Retrieve the MACONTROLINSTANCE_SWITCH struct that we allocated when we
					// created the Parameter Edit dialog. The pointer to this array was stored
					// in the Parameter Edit dialog's DWL_USER field
					hl = (HLOCAL)GetWindowLong(hwnd, DWL_USER);

					// Close the dialog with SUCCESS
					EndDialog(hwnd, 0);

					// Free the memory
					LocalFree(hl);

					// Indicate that the "Parameter Edit" dialog is no longer open
					ghdlgControl = 0;

//					break;
				}
			}
		}

//		break;
    }

    return (FALSE);
}
