//==========================================================================
// mameter.c
//
// Manages the "Parameter Edit" (IDD_PARAMEDIT) dialog for audio controls that
// are of the meter (MIXERCONTROL_CT_CLASS_METER) class. This displays the
// settings of the selected audio control for the selected audio line of the
// currently open Mixer device.
//
// This dialog is opened when the user clicks the "Value" button for a meter
// type of audio control in the Main Window's listbox full of audio control
// names.
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
// PURPOSE.
//
// Copyright (C) 1993 - 1997  Microsoft Corporation.  All Rights Reserved.
// Modified 2001 by Jeff Glatt
//==========================================================================

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>

#include "resource.h"
#include "mixapp.h"



// To fetch and set the values of all the uniform (master) and individual channel
// faders for the currently selected parameter
typedef struct tMACONTROLINSTANCE_METER
{
	BOOL						fSigned;
	DWORD						dwRange;
	MIXERCONTROLDETAILS_SIGNED	pmxcd_s[];
} MACONTROLINSTANCE_METER, *PMACONTROLINSTANCE_METER;





// ************************* changeMeterLine() *************************
// Called when our Main Window is notified that a line in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_LINE_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD dwLineID:		ID of the line whose state has changed.
//
// RETURNS:
//		1 if the line is active, 0 if inactive.

BOOL changeMeterLine(HWND hwnd, HMIXER hmx, DWORD dwLineID)
{
	HWND			htxt;
	MMRESULT		err;
	BOOL			fActive;

	// Is the line that was changed the same as the line to which our currently
	// displayed parameter belongs?
//	if (CurrMixerLine.dwLineID != dwLineID)
//	{
		// Uh oh. The Mixer is informing us of a change to another line (ie, not
		// the one that the user has currently selected in the main window).

		// Just update for the currently selected line
//		dwLineID = CurrMixerLine.dwLineID;
//	}

	// Get info on the line that changed
	CurrMixerLine.cbStruct = sizeof(MIXERLINE);

	if ((err = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &CurrMixerLine, MIXER_GETLINEINFOF_LINEID)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on line ID #%u!", err, CurrMixerLine.dwLineID);
		return(FALSE);
	}

	// Get the current state of the line that changed -- active or inactive
	fActive       = (0 != (MIXERLINE_LINEF_ACTIVE & CurrMixerLine.fdwLine));

	// Refresh the "Line Info"
	htxt = GetDlgItem(hwnd, IDD_MACONTROL_TXT_LINEINFO);
	doWindowText(htxt, "(%s), '%s', %s, %s",
					(0 != (MIXERLINE_LINEF_SOURCE & CurrMixerLine.fdwLine)) ? (LPSTR)"src" : (LPSTR)"DST",
					(LPSTR)CurrMixerLine.szShortName,
					fActive ? (LPSTR)"ACTIVE" : (LPSTR)"inactive",
					(0 != (MIXERLINE_LINEF_DISCONNECTED & CurrMixerLine.fdwLine)) ? (LPSTR)"DISCONNECTED" : (LPSTR)"connected");

	return(fActive);
}





// ************************* changeMeterParameter() *************************
// Called when our Main Window is notified that a parameter in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_CONTROL_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD controlID:	ID of the parameter whose state has changed.
//
// RETURNS:
//		0 if success, or non-zero if an error # from the Mixer API.

MMRESULT changeMeterParameter(HWND hwnd, HMIXER hmx, DWORD dwControlID)
{
	MMRESULT						err;
	MACONTROLINSTANCE_METER			*pmaci_meter;
	MIXERCONTROLDETAILS_SIGNED		*pmxcd_s;
	UINT							cChannels, cMultipleItems;
	UINT							uIndex;
	MIXERCONTROLDETAILS				mixerControlDetails;
	HWND							hwndFocus;
	HWND							hwndChild;
	BOOL							fSigned;
	DWORD							dwRange, dwValue;
	int								nValue;

	// Get the meter with the focus
	if (!(hwndFocus = GetFocus()))
	{
		// If no meter has the focus, assume the first of the individual parameter meters
		hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
	}
	else
	{
		// Get the ID of the meter with the focus
		uIndex = GetDlgCtrlID(hwndFocus);

		// If not one of the meters, then assume the first channel meter. It may
		// be that this was called as a result of the Mixer device informing us
		// of a change to our parameter, not because the user selected and moved
		// some meter
		if (uIndex < IDD_MACONTROL_MULTICHANNEL_BASE)
		{
			hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
		}
	}

	// Retrieve the array of MIXERCONTROLDETAILS_SIGNED structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_meter = (MACONTROLINSTANCE_METER *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_s = &pmaci_meter->pmxcd_s[0];
	fSigned = pmaci_meter->fSigned;
	dwRange = pmaci_meter->dwRange;

	// ================== Update the individual item meters =================

	cChannels = (UINT)CurrMixerLine.cChannels;
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

    // Get the current values of each individual meter of each channels. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;							// This is the # of channels
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;	// The # of items per channel
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_SIGNED);
	mixerControlDetails.paDetails = pmxcd_s;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	cMultipleItems = 1;
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Determine the total # of individual meters for all items
	cChannels *= cMultipleItems;

	// Start with the first meter
	uIndex = 0;

	while (cChannels--)
	{
		// Scale the next meter's value to a scrollbar knob position
		if (fSigned)
		{
			dwValue = (DWORD)(pmxcd_s[uIndex].lValue - CurrParameter.Bounds.lMinimum);
		}
		else
		{
			dwValue = (DWORD)pmxcd_s[uIndex].lValue;
			dwValue -= CurrParameter.Bounds.dwMinimum;
		}
		nValue = MulDiv(dwValue, 32767, dwRange);

		// Get the next meter
		hwndChild = GetDlgItem(hwnd, uIndex + IDD_MACONTROL_MULTICHANNEL_BASE);

		// Windows updates a scrollbar even when its knob position does not change.
		// This may cause unnecessary flickering, so don't update if the scrollbar
		// if its knob is already in the desired position
		if ((32767 - nValue) != GetScrollPos(hwndChild, SB_CTL))
		{
			// Note that we invert the scrollbar's knob position since an audio meter
			// would increase approaching the top (opposite of Windows scrollbar logic)
			SetScrollPos(hwndChild, SB_CTL, 32767 - nValue, TRUE);
		}

		// If this is the scrollbar for the currently selected meter, print out the value
		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			if (fSigned)
			{
				doWindowText(hwndChild, "mapped=%d, lValue=%ld", nValue, pmxcd_s[uIndex].lValue);
			}
			else
			{
				doWindowText(hwndChild, "mapped=%d, dwValue=%lu", nValue, pmxcd_s[uIndex].lValue);
			}
		}

		// Next scrollbar window ID
		uIndex++;
	}

	// ================== Update the uniform (ie, master) meters =================

	// Get info upon each item's master meter value. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = 1;										// Just get the master meter values -- not all individual ones
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;		// This is how many master meters
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_SIGNED);
	mixerControlDetails.paDetails = pmxcd_s;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	while (cMultipleItems--)
	{
		if (fSigned)
		{
			dwValue = (DWORD)(pmxcd_s[cMultipleItems].lValue - CurrParameter.Bounds.lMinimum);
		}
		else
		{
			dwValue = (DWORD)pmxcd_s[cMultipleItems].lValue;
			dwValue -= CurrParameter.Bounds.dwMinimum;
		}
		nValue = MulDiv(dwValue, 32767, dwRange);

		hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems);

		if ((32767 - nValue) != GetScrollPos(hwndChild, SB_CTL))
			SetScrollPos(hwndChild, SB_CTL, 32767 - nValue, TRUE);

		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			if (fSigned)
			{
				doWindowText(hwndChild, "mapped=%d, lValue=%ld", nValue, pmxcd_s[cMultipleItems].lValue);
			}
			else
			{
				doWindowText(hwndChild, "mapped=%d, dwValue=%lu", nValue, pmxcd_s[cMultipleItems].lValue);
			}
		}
	}

	// Success
	return(0);
}





// ************************** initMeterDlg() **************************
// Called by meterDlgProc()'s WM_INITDIALOG to initialize the "Parameter
// Edit" (IDD_PARAMEDIT) dialog prior to displaying it for the first
// time.
//
// NOTE: The global 'CurrParameter' MIXERCONTROL struct must already be
// filled in with the current parameter, and 'CurrMixerLine' MIXERLINE
// struct must be filled in with its line.
//
// ARGS:
//		HWND hwnd:					Handle to "Parameter Edit" dialog.
//
//		LPMACONTROLINSTANCE pmaci:	Pointer to MACONTROLINSTANCE struct
//									filled in with the values of the
//									parameter being displayed.
//
// RETURNS:
//		0 if success, or non-zero if an error (-4 if not a type of meter
//		understood by this app, -1 if memory error, -5 if a window
//		can't be created, or other error # from the Mixer API).

MMRESULT initMeterDlg(HWND hwnd)
{
#define FSB_DEF_STYLE   (WS_VISIBLE | WS_CHILD | SBS_VERT | WS_TABSTOP)

	static const TCHAR			szScrollBar[] = TEXT("scrollbar");

	MACONTROLINSTANCE_METER		*pmaci_meter;
	HWND						hwndChild;
	UINT						u, v;
	RECT						rc;
	int							cxvsb;
	UINT						cChannels, cMultipleItems;
	UINT						uIndex;
	DWORD						dwRange;
	BOOL						fSigned;

	// Check that it's a legitimate parameter type for the meter class
	switch (CurrParameter.dwControlType)
	{
		case MIXERCONTROL_CONTROLTYPE_SIGNEDMETER:
		case MIXERCONTROL_CONTROLTYPE_PEAKMETER:
		case MIXERCONTROL_CONTROLTYPE_UNSIGNEDMETER:
			break;

		default:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Invalid meter type: 0x%.08X!", CurrParameter.dwControlType);
			return((MMRESULT)-4);
		}
	}

	// Set a flag whether its value is signed or unsigned
	fSigned = (MIXERCONTROL_CT_UNITS_SIGNED == (MIXERCONTROL_CT_UNITS_MASK & CurrParameter.dwControlType));

	// Set the dialog's title to the name of the parameter being edited
	SetWindowText(hwnd, &CurrParameter.szName[0]);

	// Display the minimum and maximum limits
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_BOUNDS);
	if (fSigned)
	{
		doWindowText(hwndChild, "lMinimum=%ld, lMaximum=%ld", CurrParameter.Bounds.lMinimum, CurrParameter.Bounds.lMaximum);
        dwRange = (CurrParameter.Bounds.lMaximum - CurrParameter.Bounds.lMinimum);
    }
    else
    {
        doWindowText(hwndChild, "dwMinimum=%lu, dwMaximum=%lu", CurrParameter.Bounds.dwMinimum, CurrParameter.Bounds.dwMaximum);
        dwRange = (CurrParameter.Bounds.dwMaximum - CurrParameter.Bounds.dwMinimum);
    }

	// Display the steps
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_METRICS);
	doWindowText(hwndChild, "dwRange=%lu", dwRange);

	// Determine how many arrays we'll need, based upon how many items there are,
	// and how many channels are in each item

	// Assume not mono
	cChannels = (UINT)CurrMixerLine.cChannels;

	// If a uniform parameter, then we have only a mono adjustment
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

	// Assume only 1 item
	cMultipleItems = 1;

	// If multiple items, then get how many items
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Get a MACONTROLINSTANCE_METER struct with an array of MIXERCONTROLDETAILS_UNSIGNED
	// structs for this parameter. The number of array structs are cMultipleItems * cChannels
	// so that we can set/retrieve all individual items on all channels simultaneously
	v = sizeof(MACONTROLINSTANCE_METER) + (cChannels * cMultipleItems * sizeof(MIXERCONTROLDETAILS_SIGNED));
	if (!(pmaci_meter = (MACONTROLINSTANCE_METER *)LocalAlloc(LPTR, v)))
 	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't get memory for sliders!");
 		return((MMRESULT)-1);
 	}

	// Save the pointer in the dialog's DWL_USER field so that other functions can retrieve it
	SetWindowLong(hwnd, DWL_USER, (LPARAM)pmaci_meter);

	// Save these values so that setMeterParameter() and changeMeterParameter() can access them
	pmaci_meter->fSigned = fSigned;
	pmaci_meter->dwRange = dwRange;

	// Figure out how wide to make each meter based upon how many meters we need
	// to squeeze into the IDC_PARAMEDIT_MULTICHANNEL groupbox
	cxvsb = GetSystemMetrics(SM_CXVSCROLL);

	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_MULTICHANNEL);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.right = rc.left + cxvsb;

	// Create the meters

	// ================== Update the individual item meters =================

	// Start with a unique window ID
	uIndex = IDD_MACONTROL_MULTICHANNEL_BASE;

	// For each channel...
	for (u = 0; u < cChannels; u++)
	{
		// and each item on this channel...
		for (v = 0; v < cMultipleItems; v++)
		{
			// Create a graphical meter, and then increment the window ID #
			if (!(hwndChild = CreateWindow(szScrollBar, 0, FSB_DEF_STYLE,
								rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
								hwnd, (HMENU)(uIndex++),
								MyInstance, NULL)))
			{
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create meter!");
				return((MMRESULT)-5);
			}

			// Set the range
			SetScrollRange(hwndChild, SB_CTL, 0, 32767, FALSE);

			// Move over for next item's meter
			rc.left  += cxvsb + 4;
			rc.right += cxvsb + 4;
		}

		// Add separation between this meter and the next channel's meter
		rc.left  += cxvsb;
		rc.right += cxvsb;
	}

	// ======================= Update the master meters ======================

	// Figure out how wide to make each meter based upon how many meters we have
	// to squeeze into the IDC_PARAMEDIT_UNIFORM groupbox
	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_UNIFORM);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.right = rc.left + cxvsb;

	// Create the meters
	for (v = 0; v < cMultipleItems; v++)
	{
		if (!(hwndChild = CreateWindow(szScrollBar, 0, FSB_DEF_STYLE,
							rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
							hwnd, (HMENU)(IDD_MACONTROL_UNIFORM_BASE + v),
							MyInstance, NULL)))
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create fader!");
			return((MMRESULT)-5);
		}

		SetScrollRange(hwndChild, SB_CTL, 0, 32767, FALSE);

		rc.left  += cxvsb + 4;
		rc.right += cxvsb + 4;
	}

	// =========================================================================

	// Send MM_MIXM_LINE_CHANGE and MM_MIXM_CONTROL_CHANGE messages to
	// IDD_PARAMEDIT dialog so that it updates its window per the
	// current settings of the current line and audio parameter. (ie,
	// Just like its window procedure would do if the operating system
	// sent it these messages)
	SendMessage(hwnd, MM_MIXM_LINE_CHANGE, (WPARAM)MixerHandle, CurrMixerLine.dwLineID);
	SendMessage(hwnd, MM_MIXM_CONTROL_CHANGE, (WPARAM)MixerHandle, CurrParameter.dwControlID);

	// Success
	return(0);
}





// *************************** meterDlgProc() ****************************
// Dialog procedure for the "Parameter Edit" (IDD_PARAMEDIT) dialog when
// it is displaying the settings of a meter (MIXERCONTROL_CT_CLASS_METER)
// type of audio parameter.
//
// This dialog is opened when the user clicks the "Value" button for a
// meter type of control in the main window's dialog's listbox full of
// parameter names.
//
//	ARGS:
//		HWND hwnd:		Handle to DLG_MIXAPP_CONTROL dialog.
//
//		UINT uMsg:		The message type.
//
//		WPARAM wParam:	Depends upon uMsg.
//
//		LPARAM lParam:	Depends upon uMsg.
//
//	RETURNS:
//		Typically, TRUE if the message is handled, or FALSE if not.

BOOL APIENTRY meterDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	static  BOOL	fTimerActive;

	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			MMRESULT	err;

			//
			fTimerActive = 0;

			// Create some child control windows, such as meters or checkmarks, which
			// will display the values of the channel masters and individual channels
			// for this audio parameter
			if ((err = initMeterDlg(hwnd)))

			// If an error, close the dialog
			{
				EndDialog(hwnd, err);
			}

			// If success, save the handle of this dialog in a global so that we'll
			// know it's open in case the operating system tells us that this audio
			// parameter has been updated
			else
				ghdlgControl = hwnd;

			return(TRUE);
		}

		// The operating system is notifying us that some line of the mixer we
		// currently have open is changed. We may need to now fetch that line's
		// new settings and update our window (assuming that we're currently
		// displaying that line's settings)
		case MM_MIXM_LINE_CHANGE:
		{
			if (changeMeterLine(hwnd, (HMIXER)wParam, lParam))
			{
				if (!fTimerActive) fTimerActive = (0 != SetTimer(hwnd, 1, 100, NULL));
			}
			else if (fTimerActive)
			{
				KillTimer(hwnd, 1);
				fTimerActive = 0;
			}
			return(TRUE);
		}

		case WM_TIMER:

		// The operating system is notifying us that some line of the mixer we
		// currently have open is changed. We may need to now fetch that line's
		// new settings and update our window (assuming that we're currently
		// displaying that line's settings)
		case MM_MIXM_CONTROL_CHANGE:
		{
			changeMeterParameter(hwnd, (HMIXER)wParam, lParam);
			return(TRUE);
		}

		case WM_COMMAND:
		{
			// If OK button, or window close button, close the dialog
			// and free memory
			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDCANCEL:
				{
					HLOCAL		hl;

					if (fTimerActive)
					{
						KillTimer(hwnd, 1);
						fTimerActive = 0;
					}

					// Retrieve the MACONTROLINSTANCE_METER struct that we allocated when
					// we created the Parameter Edit dialog. The pointer to this array was stored
					// in the Parameter Edit dialog's DWL_USER field
					hl = (HLOCAL)GetWindowLong(hwnd, DWL_USER);

					// Close the dialog with SUCCESS
					EndDialog(hwnd, 0);

					// Free the memory
					LocalFree(hl);

					// Indicate that the "Parameter Edit" dialog is no longer open
					ghdlgControl = 0;

//					break;
				}
			}
//			break;
		}
	}

	return(FALSE);
}
