//==========================================================================
// malist.c
//
// Manages the "Parameter Edit" (IDD_PARAMEDIT) dialog for audio controls that
// are of the list (MIXERCONTROL_CT_CLASS_LIST) class. This displays the
// settings of the selected audio control for the selected audio line of the
// currently open Mixer device.
//
// This dialog is opened when the user clicks the "Value" button for a list
// type of audio control in the Main Window's listbox full of audio control
// names.
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
// PURPOSE.
//
// Copyright (C) 1993 - 1997  Microsoft Corporation.  All Rights Reserved.
// Modified 2001 by Jeff Glatt
//==========================================================================

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>

#include "resource.h"
#include "mixapp.h"



//
typedef struct tMACONTROLINSTANCE_LIST
{
	BOOL						fSingleSelect;
	MIXERCONTROLDETAILS_BOOLEAN	pmxcd_f[];
} MACONTROLINSTANCE_LIST, *PMACONTROLINSTANCE_LIST;





// ************************* changeListLine() *************************
// Called when our Main Window is notified that a line in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_LINE_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD dwLineID:		ID of the line whose state has changed.
//
// RETURNS:
//		1 if the line is active, 0 if inactive.

BOOL changeListLine(HWND hwnd, HMIXER hmx, DWORD dwLineID)
{
	HWND			htxt;
	MMRESULT		err;
	BOOL			fActive;

	// Is the line that was changed the same as the line to which our currently
	// displayed parameter belongs?
//	if (CurrMixerLine.dwLineID != dwLineID)
//	{
		// Uh oh. The Mixer is informing us of a change to another line (ie, not
		// the one that the user has currently selected in the main window).

		// Just update for the currently selected line
//		dwLineID = CurrMixerLine.dwLineID;
//	}

	// Get info on the line that changed
	CurrMixerLine.cbStruct = sizeof(MIXERLINE);

	if ((err = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &CurrMixerLine, MIXER_GETLINEINFOF_LINEID)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on line ID #%u!", err, CurrMixerLine.dwLineID);
		return(FALSE);
	}

	// Get the current state of the line that changed -- active or inactive
	fActive       = (0 != (MIXERLINE_LINEF_ACTIVE & CurrMixerLine.fdwLine));

	// Refresh the "Line Info"
	htxt = GetDlgItem(hwnd, IDD_MACONTROL_TXT_LINEINFO);
	doWindowText(htxt, "(%s), '%s', %s, %s",
					(0 != (MIXERLINE_LINEF_SOURCE & CurrMixerLine.fdwLine)) ? (LPSTR)"src" : (LPSTR)"DST",
					(LPSTR)CurrMixerLine.szShortName,
					fActive ? (LPSTR)"ACTIVE" : (LPSTR)"inactive",
					(0 != (MIXERLINE_LINEF_DISCONNECTED & CurrMixerLine.fdwLine)) ? (LPSTR)"DISCONNECTED" : (LPSTR)"connected");

	return(fActive);
}





// ************************* changeListParameter() *************************
// Called when our Main Window is notified that a parameter in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_CONTROL_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD controlID:	ID of the parameter whose state has changed.
//
// RETURNS:
//		0 if success, or non-zero if an error # from the Mixer API.

MMRESULT changeListParameter(HWND hwnd, HMIXER hmx, DWORD dwControlID)
{
	MMRESULT						err;
	PMACONTROLINSTANCE_LIST			pmaci_list;
	PMIXERCONTROLDETAILS_BOOLEAN    pmxcd_f;
	BOOL							fValue;
	UINT							cChannels, cMultipleItems;
	UINT							uIndex;
	MIXERCONTROLDETAILS				mixerControlDetails;
	HWND							hwndFocus;
	HWND							hwndChild;

	// Get the list item with the focus
	if (!(hwndFocus = GetFocus()))
	{
		// If no item has the focus, assume the first of the individual parameter list items
		hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
	}
	else
	{
		// Get the ID of the list item with the focus
		uIndex = GetDlgCtrlID(hwndFocus);

		// If not one of the list items, then assume the first channel list item. It may
		// be that this was called as a result of the Mixer device informing us
		// of a change to our parameter, not because the user selected some list item
		if (uIndex < IDD_MACONTROL_MULTICHANNEL_BASE)
		{
			hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
		}
	}

	// Retrieve the array of MIXERCONTROLDETAILS_BOOLEAN structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_list = (MACONTROLINSTANCE_LIST *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_f = &pmaci_list->pmxcd_f[0];

	// ================== Update the individual item lists =================

	cChannels = (UINT)CurrMixerLine.cChannels;
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

    // Get the current values of each individual list item of each channels. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;							// This is the # of channels
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;	// The # of items per channel
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_BOOLEAN);
	mixerControlDetails.paDetails = pmxcd_f;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}
	
	cMultipleItems = 1;
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Determine the total # of individual faders for all items
	cChannels *= cMultipleItems;

	// Start with the first list item
	uIndex = 0;

	while (cChannels--)
	{
		fValue = (BOOL)pmxcd_f[uIndex].fValue;

		hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE + uIndex);

		// If this is the currently selected list item, print out the value
		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			doWindowText(hwndChild, "fValue=%lu", fValue);
		}

		// If its state is not already set as desired, set its state
		if (fValue != (BOOL)IsDlgButtonChecked(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE + uIndex))
		{
			CheckDlgButton(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE + uIndex, fValue);
		}

		// Next list item window ID
		uIndex++;
    }

	// ================== Update the uniform (ie, master) list items =================

	// Get info upon each item's master list value. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = 1;										// Just get the master fader values -- not all individual ones
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;		// This is how many master faders
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_BOOLEAN);
	mixerControlDetails.paDetails = pmxcd_f;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	while (cMultipleItems--)
	{
		fValue = (BOOL)pmxcd_f[cMultipleItems].fValue;

		hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems);

		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			doWindowText(hwndChild, "fValue=%lu", fValue);
		}

		// If it's state is not already set as desired, set its state
		if (fValue != (BOOL)IsDlgButtonChecked(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems))
		{
			CheckDlgButton(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems, fValue);
		}
	}

	// Success
	return(0);
}





// *************************** setListParameter() ***************************
// Called when a graphical list item is selected by the user to set the value
// of some audio parameter of list type.
//
// ARGS:
//		HWND hwnd:		Handle to Parameter Edit dialog for list type of
//						parameter.
//
//      HWND hcheck:	Handle to listbox graphical control.
//
// RETURNS:
//		0 if success, or non-zero if an error # from the Mixer API.

MMRESULT setListParameter(HWND hwnd, HWND hcheck)
{
	PMACONTROLINSTANCE_LIST			pmaci_list;
	PMIXERCONTROLDETAILS_BOOLEAN	pmxcd_f;
	MMRESULT						err;
	BOOL							fValue;
	UINT							cChannels;
	UINT							uIndex;
	MIXERCONTROLDETAILS				mixerControlDetails;

	// Retrieve the array of MIXERCONTROLDETAILS_BOOLEAN structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_list = (MACONTROLINSTANCE_LIST *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_f = &pmaci_list->pmxcd_f[0];

	uIndex = GetDlgCtrlID(hcheck);

	// If single-select, then its value is TRUE (ie, selected)
	if (pmaci_list->fSingleSelect)
		fValue = TRUE;

	// If multiple-select, then we toggle its value
	else
		fValue = (0 == IsDlgButtonChecked(hwnd, uIndex));

	if (uIndex < IDD_MACONTROL_UNIFORM_BASE)
	{
		cChannels = (UINT)CurrMixerLine.cChannels;
		if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
			cChannels = 1;

		uIndex -= IDD_MACONTROL_MULTICHANNEL_BASE;
	}
	else
	{
		cChannels = 1;

		uIndex -= IDD_MACONTROL_UNIFORM_BASE;
	}

	// Fetch the value of the list item
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_BOOLEAN);
	mixerControlDetails.paDetails = pmxcd_f;

	if ((err = mixerGetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	// If single-select, then we need to deselect all of the other items on this channel
	if (pmaci_list->fSingleSelect)
	{
		UINT	u, v, cMultipleItems;

		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

		u = uIndex / (cChannels * cMultipleItems);

		for (v = 0; v < cMultipleItems; v++)
		{
			pmxcd_f[(u * cMultipleItems) + v].fValue = FALSE;
		}
	}

	// Set the selected item's value as desired
	pmxcd_f[uIndex].fValue = fValue;

	// Send this value to the Mixer
	if ((err = mixerSetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_SETCONTROLDETAILSF_VALUE)))
    {
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u setting value of parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	// Set the list item's state
	CheckDlgButton(hwnd, GetDlgCtrlID(hcheck), (BOOL)pmxcd_f[uIndex].fValue);

	// Success
	return(0);
}





// ************************** initFaderDlg() **************************
// Called by listDlgProc()'s WM_INITDIALOG to initialize the "Parameter
// Edit" (IDD_PARAMEDIT) dialog prior to displaying it for the first
// time.
//
// NOTE: The global 'CurrParameter' MIXERCONTROL struct must already be
// filled in with the current parameter, and 'CurrMixerLine' MIXERLINE
// struct must be filled in with its line.
//
// ARGS:
//		HWND hwnd:					Handle to "Parameter Edit" dialog.
//
//		LPMACONTROLINSTANCE pmaci:	Pointer to MACONTROLINSTANCE struct
//									filled in with the values of the
//									parameter being displayed.
//
// RETURNS:
//		0 if success, or non-zero if an error (-4 if not a type of list
//		understood by this app, -1 if memory error, -5 if a window
//		can't be created, or other error # from the Mixer API).

MMRESULT initListDlg(HWND hwnd)
{
    #define FCB_DEF_STYLE   (WS_VISIBLE | WS_CHILD | BS_CHECKBOX | WS_TABSTOP)

    static const TCHAR				szButton[]  = TEXT("button");

	MACONTROLINSTANCE_LIST			*pmaci_list;
	MIXERCONTROLDETAILS_LISTTEXT	*pmxcd_lt;
	MMRESULT						err;
	UINT							u, v;
	HWND							hwndChild;
	RECT							rc;
	int								cycap;
	UINT							cChannels, cMultipleItems;
	UINT							uIndex;
	BOOL							fSingleSelect;
	MIXERCONTROLDETAILS				mixerControlDetails;

	// Check that it's a legitimate parameter type for the list class
	switch (CurrParameter.dwControlType)
	{
		case MIXERCONTROL_CONTROLTYPE_SINGLESELECT:
		case MIXERCONTROL_CONTROLTYPE_MUX:
		case MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT:
		case MIXERCONTROL_CONTROLTYPE_MIXER:
			break;

		default:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Invalid list type: 0x%.08X!", CurrParameter.dwControlType);
			return((MMRESULT)-4);
		}
	}

	// Set the dialog's title to the name of the parameter being edited
	SetWindowText(hwnd, &CurrParameter.szName[0]);

	// Set a flag whether single or multiple select
	fSingleSelect = (MIXERCONTROL_CT_SC_LIST_SINGLE == (MIXERCONTROL_CT_SUBCLASS_MASK & CurrParameter.dwControlType));

	// Display the minimum and maximum limits
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_BOUNDS);
	doWindowText(hwndChild, "dwMinimum=%lu, dwMaximum=%lu", CurrParameter.Bounds.dwMinimum, CurrParameter.Bounds.dwMaximum);

	// The steps are not applicable for a list class, so hide that display
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_METRICS);
	ShowWindow(hwndChild, SW_HIDE);

	// Determine how many arrays we'll need, based upon how many items there are,
	// and how many channels are in each item

	// Assume not mono
	cChannels = (UINT)CurrMixerLine.cChannels;

	// If a uniform parameter, then we have only a mono adjustment
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

	// Assume only 1 item
	cMultipleItems = 1;

	// If multiple items, then get how many items
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Get an array of MIXERCONTROLDETAILS_LISTTEXT structs for this parameter. The number of
	// array structs are cMultipleItems * cChannels so that we can retrieve the text (labels)
	// of all individual items on all channels simultaneously
	if (!(pmxcd_lt = (MIXERCONTROLDETAILS_LISTTEXT *)LocalAlloc(LPTR, cChannels * cMultipleItems * sizeof(MIXERCONTROLDETAILS_LISTTEXT))))
 	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't get memory for lists!");
 		return((MMRESULT)-1);
 	}

	// Get the text labels for the items in the list
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;
	mixerControlDetails.cMultipleItems = cMultipleItems;
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_LISTTEXT);
	mixerControlDetails.paDetails = pmxcd_lt;

	if ((err = mixerGetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_LISTTEXT)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%d getting LISTTEXT for control ID #%d!", err, CurrParameter.dwControlID);
		return(err);
	}

	// Figure out how wide to make each listbox based upon how many items we need
	// to squeeze into the IDC_PARAMEDIT_MULTICHANNEL groupbox
	cycap = GetSystemMetrics(SM_CYCAPTION);

	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_MULTICHANNEL);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.bottom = rc.top + cycap;

	// Create the listboxes

	// ================== Update the individual item listboxes =================

	// Start with a unique window ID
	uIndex = IDD_MACONTROL_MULTICHANNEL_BASE;

	// For each channel...
	for (u = 0; u < cChannels; u++)
	{
		// and each item on this channel...
		for (v = 0; v < cMultipleItems; v++)
		{
			// Create a listbox, and then increment the window ID #
			if (!(hwndChild = CreateWindow(szButton, pmxcd_lt[uIndex - IDD_MACONTROL_MULTICHANNEL_BASE].szName, FCB_DEF_STYLE,
								rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
								hwnd, (HMENU)(uIndex++),
								MyInstance, NULL)))
			{
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create listbox!");
				return((MMRESULT)-5);
			}

			// Move over for next item's listbox
			rc.top  += cycap + 4;
			rc.bottom += cycap + 4;
		}

		// Add separation between this listbox and the next channel's listbox
		rc.top  += cycap;
		rc.bottom += cycap;
	}

	// ======================= Update the master lists ======================

	// Get the text labels for the items in the list
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = 1;
	mixerControlDetails.cMultipleItems = cMultipleItems;
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_LISTTEXT);
	mixerControlDetails.paDetails = pmxcd_lt;

	if ((err = mixerGetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_LISTTEXT)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%d getting LISTTEXT for control ID #%d!", err, CurrParameter.dwControlID);
		return(err);
	}

	// Figure out how wide to make each listbox based upon how many lists we have
	// to squeeze into the IDC_PARAMEDIT_UNIFORM groupbox
	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_UNIFORM);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.bottom = rc.top + cycap;

	for (v = 0; v < cMultipleItems; v++)
	{
		if (!(hwndChild = CreateWindow(szButton,
                              pmxcd_lt[v].szName, FCB_DEF_STYLE,
                              rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
                              hwnd, (HMENU)(IDD_MACONTROL_UNIFORM_BASE + v),
                              MyInstance, NULL)))
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create listbox!");
			return((MMRESULT)-5);
		}

		rc.top += cycap + 4;
		rc.bottom += cycap + 4;
	}

	// Free the LISTTEXT structures
	LocalFree((HLOCAL)pmxcd_lt);

	// Get a MACONTROLINSTANCE_LIST struct with an array of MIXERCONTROLDETAILS_BOOLEAN
	// structs for this parameter. The number of array structs are cMultipleItems * cChannels
	// so that we can set/retrieve all individual items on all channels simultaneously
	v = sizeof(MACONTROLINSTANCE_LIST) + (cChannels * cMultipleItems * sizeof(MIXERCONTROLDETAILS_BOOLEAN));
	if (!(pmaci_list = (MACONTROLINSTANCE_LIST *)LocalAlloc(LPTR, v)))
 	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't get memory for lists!");
 		return((MMRESULT)-1);
 	}

	// Save the pointer in the dialog's DWL_USER field so that other functions can retrieve it
	SetWindowLong(hwnd, DWL_USER, (LPARAM)pmaci_list);

	// =========================================================================

	// Send MM_MIXM_LINE_CHANGE and MM_MIXM_CONTROL_CHANGE messages to
	// IDD_PARAMEDIT dialog so that it updates its window per the
	// current settings of the current line and audio parameter. (ie,
	// Just like its window procedure would do if the operating system
	// sent it these messages)
	SendMessage(hwnd, MM_MIXM_LINE_CHANGE, (WPARAM)MixerHandle, CurrMixerLine.dwLineID);
	SendMessage(hwnd, MM_MIXM_CONTROL_CHANGE, (WPARAM)MixerHandle, CurrParameter.dwControlID);

	// Success
	return(0);
}





// *************************** listDlgProc() ****************************
// Dialog procedure for the "Parameter Edit" (IDD_PARAMEDIT) dialog when
// it is displaying the settings of a list (MIXERCONTROL_CT_CLASS_LIST)
// type of audio parameter.
//
// This dialog is opened when the user clicks the "Value" button for a
// list type of control in the main window's dialog's listbox full of
// parameter names.
//
//	ARGS:
//		HWND hwnd:		Handle to DLG_MIXAPP_CONTROL dialog.
//
//		UINT uMsg:		The message type.
//
//		WPARAM wParam:	Depends upon uMsg.
//
//		LPARAM lParam:	Depends upon uMsg.
//
//	RETURNS:
//		Typically, TRUE if the message is handled, or FALSE if not.

BOOL APIENTRY listDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			MMRESULT	err;

			// Create some child control windows, such as listboxes or checkmarks, which
			// will display the values of the channel masters and individual channels
			// for this audio parameter
			if ((err = initListDlg(hwnd)))

			// If an error, close the dialog
			{
				EndDialog(hwnd, err);
			}

			// If success, save the handle of this dialog in a global so that we'll
			// know it's open in case the operating system tells us that this audio
			// parameter has been updated
			else
				ghdlgControl = hwnd;

			return(TRUE);
		}

		// The operating system is notifying us that some line of the mixer we
		// currently have open is changed. We may need to now fetch that line's
		// new settings and update our window (assuming that we're currently
		// displaying that line's settings)
		case MM_MIXM_LINE_CHANGE:
		{
			changeListLine(hwnd, (HMIXER)wParam, lParam);
			return(TRUE);
		}

		// The operating system is notifying us that some audio parameter of the
		// mixer we currently have open is changed. We may need to now fetch
		// that parameter's new settings and update our window (assuming that
		// we're currently displaying that parameter's settings)
		case MM_MIXM_CONTROL_CHANGE:
		{
			changeListParameter(hwnd, (HMIXER)wParam, lParam);
			return (TRUE);
		}

		case WM_COMMAND:
		{
			// If a selection made in one of the listboxes, handle it
			if (IDC_PARAMEDIT_MULTICHANNEL <= GET_WM_COMMAND_ID(wParam, lParam))
			{
				setListParameter(hwnd, GET_WM_COMMAND_HWND(wParam, lParam));
				return(TRUE);
			}

			// If OK button, or window close button, close the dialog
			// and free memory
			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDCANCEL:
				{
					HLOCAL		hl;

					// Retrieve the MACONTROLINSTANCE_LIST struct that we allocated when we
					// created the Parameter Edit dialog. The pointer to this array was stored
					// in the Parameter Edit dialog's DWL_USER field
					hl = (HLOCAL)GetWindowLong(hwnd, DWL_USER);

					// Close the dialog with SUCCESS
					EndDialog(hwnd, 0);

					// Free the memory
					LocalFree(hl);

					// Indicate that the "Parameter Edit" dialog is no longer open
					ghdlgControl = 0;

//					break;
				}
			}
//			break;
		}
	}

	return (FALSE);
}
