//==========================================================================
// mafader.c
//
// Manages the "Parameter Edit" (IDD_PARAMEDIT) dialog for audio controls that
// are of the fader (MIXERCONTROL_CT_CLASS_FADER) class. This displays the
// settings of the selected audio control for the selected audio line of the
// currently open Mixer device.
//
// This dialog is opened when the user clicks the "Value" button for a fader
// type of audio control in the Main Window's listbox full of audio control
// names.
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
// PURPOSE.
//
// Copyright (C) 1993 - 1997  Microsoft Corporation.  All Rights Reserved.
// Modified 2001 by Jeff Glatt
//==========================================================================

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>

#include "resource.h"
#include "mixapp.h"



// To fetch and set the values of all the uniform (master) and individual channel
// faders for the currently selected parameter
typedef struct tMACONTROLINSTANCE_FADER
{
	int								nRange;
	int                             nPageInc;
	MIXERCONTROLDETAILS_UNSIGNED	pmxcd_u[];
} MACONTROLINSTANCE_FADER, *PMACONTROLINSTANCE_FADER;





// ************************* changeFaderLine() *************************
// Called when our Main Window is notified that a line in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_LINE_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD dwLineID:		ID of the line whose state has changed.
//
// RETURNS:
//		1 if the line is active, 0 if inactive.

BOOL changeFaderLine(HWND hwnd, HMIXER hmx, DWORD dwLineID)
{
	HWND			htxt;
	MMRESULT		err;
	BOOL			fActive;

	// Is the line that was changed the same as the line to which our currently
	// displayed parameter belongs?
//	if (CurrMixerLine.dwLineID != dwLineID)
//	{
		// Uh oh. The Mixer is informing us of a change to another line (ie, not
		// the one that the user has currently selected in the main window).

		// Just update for the currently selected line
//		dwLineID = CurrMixerLine.dwLineID;
//	}

	// Get info on the line that changed
	CurrMixerLine.cbStruct = sizeof(MIXERLINE);

	if ((err = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &CurrMixerLine, MIXER_GETLINEINFOF_LINEID)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on line ID #%u!", err, CurrMixerLine.dwLineID);
		return(FALSE);
	}

	// Get the current state of the line that changed -- active or inactive
	fActive       = (0 != (MIXERLINE_LINEF_ACTIVE & CurrMixerLine.fdwLine));

	// Refresh the "Line Info"
	htxt = GetDlgItem(hwnd, IDD_MACONTROL_TXT_LINEINFO);
	doWindowText(htxt, "(%s), '%s', %s, %s",
					(0 != (MIXERLINE_LINEF_SOURCE & CurrMixerLine.fdwLine)) ? (LPSTR)"src" : (LPSTR)"DST",
					(LPSTR)CurrMixerLine.szShortName,
					fActive ? (LPSTR)"ACTIVE" : (LPSTR)"inactive",
					(0 != (MIXERLINE_LINEF_DISCONNECTED & CurrMixerLine.fdwLine)) ? (LPSTR)"DISCONNECTED" : (LPSTR)"connected");

	return(fActive);
}





// ************************* changeFaderParameter() *************************
// Called when our Main Window is notified that a parameter in our currently
// open Mixer has changed its state.
//
// ARGS:
//		HWND hwnd:			Handle of window that receives the
//							MM_MIXM_CONTROL_CHANGE message from the Mixer.
//
//		HMIXER hmx:			Handle of the Mixer sending the message.
//
//		DWORD controlID:	ID of the parameter whose state has changed.
//
// RETURNS:
//		0 if success, or non-zero if an error # from the Mixer API.

MMRESULT changeFaderParameter(HWND hwnd, HMIXER hmx, DWORD dwControlID)
{
	MACONTROLINSTANCE_FADER			*pmaci_fader;
	MIXERCONTROLDETAILS_UNSIGNED	*pmxcd_u;
	MMRESULT						err;
	int								nRange, nValue;
	UINT							cChannels, cMultipleItems;
	UINT							uIndex;
	MIXERCONTROLDETAILS				mixerControlDetails;
	HWND							hwndFocus;
	HWND							hwndChild;

	// Get the fader with the focus
	if (!(hwndFocus = GetFocus()))
	{
		// If no fader has the focus, assume the first of the individual parameter faders
		hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
	}
	else
	{
		// Get the ID of the fader with the focus
		uIndex = GetDlgCtrlID(hwndFocus);

		// If not one of the faders, then assume the first channel fader. It may
		// be that this was called as a result of the Mixer device informing us
		// of a change to our parameter, not because the user selected and moved
		// some fader
		if (uIndex < IDD_MACONTROL_MULTICHANNEL_BASE)
		{
			hwndFocus = GetDlgItem(hwnd, IDD_MACONTROL_MULTICHANNEL_BASE);
		}
	}

	// Retrieve the array of MIXERCONTROLDETAILS_UNSIGNED structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_fader = (MACONTROLINSTANCE_FADER *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_u = &pmaci_fader->pmxcd_u[0];
	nRange = pmaci_fader->nRange;

	// ================== Update the individual item faders =================

	cChannels = (UINT)CurrMixerLine.cChannels;
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

    // Get the current values of each individual fader of each channels. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;							// This is the # of channels
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;	// The # of items per channel
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_UNSIGNED);
	mixerControlDetails.paDetails = pmxcd_u;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	cMultipleItems = 1;
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Determine the total # of individual faders for all items
	cChannels *= cMultipleItems;

	// Start with the first fader
	uIndex = 0;

	while (cChannels--)
	{
		// Scale the next fader's value to a scrollbar knob position
		nValue = MulDiv(pmxcd_u[uIndex].dwValue, nRange, 0xFFFF);

		// Get the next scrollbar
		hwndChild = GetDlgItem(hwnd, uIndex + IDD_MACONTROL_MULTICHANNEL_BASE);

		// Windows updates a scrollbar even when its knob position does not change.
		// This may cause unnecessary flickering, so don't update if the scrollbar
		// if its knob is already in the desired position
		if ((nRange - nValue) != GetScrollPos(hwndChild, SB_CTL))
		{
			// Note that we invert the scrollbar's knob position since an audio fader
			// would increase approaching the top (opposite of Windows scrollbar logic)
			SetScrollPos(hwndChild, SB_CTL, nRange - nValue, TRUE);
		}

		// If this is the scrollbar for the currently selected fader, print out the value
		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			doWindowText(hwndChild, "mapped=%d, dwValue=%lu", nValue, pmxcd_u[uIndex].dwValue);
		}

		// Next scrollbar window ID
		uIndex++;
	}

	// ================== Update the uniform (ie, master) faders =================

	// Get info upon each item's master fader value. Get them all at once
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = 1;										// Just get the master fader values -- not all individual ones
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;		// This is how many master faders
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_UNSIGNED);
	mixerControlDetails.paDetails = pmxcd_u;

	if ((err = mixerGetControlDetails((HMIXEROBJ)hmx, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, mixerControlDetails.dwControlID);
		return(err);
	}

	while (cMultipleItems--)
	{
		nValue = MulDiv(pmxcd_u[cMultipleItems].dwValue, nRange, 0xFFFF);

		hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_UNIFORM_BASE + cMultipleItems);

		if ((nRange - nValue) != GetScrollPos(hwndChild, SB_CTL))
			SetScrollPos(hwndChild, SB_CTL, nRange - nValue, TRUE);

		if (hwndFocus == hwndChild && (hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_VALUE)))
		{
			doWindowText(hwndChild, "mapped=%d, dwValue=%lu", nValue, pmxcd_u[cMultipleItems].dwValue);
		}
	}

	// Success
	return(0);
}





// *************************** setFaderParameter() ***************************
// Called when a graphical fader is moved by the user to set the value of some
// audio parameter of fader type.
//
// ARGS:
//		HWND hwnd:		Handle to Parameter Edit dialog for fader type of
//						parameter.
//
//      HWND hsb:		Handle to scroll bar graphical control.
//
//      UINT uCode:		Action of the scroll bar.
//
//      int nPos:		Position of the scroll bar.
//
// RETURNS:
//		0 if success, or -4 if not a known nCode, or other non-zero if an error
//		# from the Mixer API.

MMRESULT setFaderParameter(HWND hwnd, HWND hsb, UINT uCode, int nPos)
{
	MACONTROLINSTANCE_FADER			*pmaci_fader;
	MIXERCONTROLDETAILS_UNSIGNED	*pmxcd_u;
	MIXERCONTROLDETAILS             mixerControlDetails;
	MMRESULT						err;
	int								nRange, nPageInc, nValue;
	UINT                            cChannels;
	UINT                            uIndex;

	// Retrieve the array of MIXERCONTROLDETAILS_UNSIGNED structs that we allocated
	// when we created the "Parameter Edit" dialog. The pointer to this array was
	// stored this dialog's DWL_USER field
	pmaci_fader = (MACONTROLINSTANCE_FADER *)GetWindowLong(hwnd, DWL_USER);
	pmxcd_u = &pmaci_fader->pmxcd_u[0];
	nRange = pmaci_fader->nRange;
	nPageInc = pmaci_fader->nPageInc;

	// Get the scroll bar's position
	nValue = GetScrollPos(hsb, SB_CTL);

	// Determine what the user did with the scroll bar, and scale the
	// final value that we'll send to the mixer
	switch (uCode)
	{
		case SB_PAGEDOWN:
		{
			nValue = (int)min(nRange, (LONG)nValue + nPageInc);
			break;
		}

		case SB_LINEDOWN:
		{
			nValue = (int)min(nRange, (LONG)nValue + 1);
			break;
		}

		case SB_PAGEUP:
		{
			nValue -= nPageInc;

			//-- fall through --//
		}

		case SB_LINEUP:
		{
			nValue = (nValue < 1) ? 0 : (nValue - 1);
			break;
		}

		case SB_TOP:
		{
			//  !!! would be a VERY BAD IDEA to go full volume !!!
			nValue = nRange / 2;
			break;
		}

		case SB_BOTTOM:
		{
			nValue = nRange;
			break;
		}

		case SB_THUMBPOSITION:
		case SB_THUMBTRACK:
		{
			nValue = nPos;
			break;
		}

		default:
		{
			return((MMRESULT)-4);
		}
	}

	// Get the window ID of the scroll bar
	uIndex = GetDlgCtrlID(hsb);

	cChannels = 1;

	// Is this one of the master faders?
	if (uIndex < IDD_MACONTROL_UNIFORM_BASE)
	{
		// Yes it is. Reference the index from IDD_MACONTROL_MULTICHANNEL_BASE
		uIndex -= IDD_MACONTROL_MULTICHANNEL_BASE;

		// If there are multiple items and this is not a UNIFORM-only control, we need to
		// fetch all of them at once with mixerGetControlDetails()
		if (!(MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl))
			cChannels = (UINT)CurrMixerLine.cChannels;
	}

	// No, it's one of the uniform (master) faders. Reference the index from IDD_MACONTROL_UNIFORM_BASE
	else
	{
		uIndex -= IDD_MACONTROL_UNIFORM_BASE;
	}

	// Fill the appropriate PMIXERCONTROLDETAILS_UNSIGNED struct (in our array for the
	// current parameter) with this audio parameter's values
	mixerControlDetails.cbStruct = sizeof(MIXERCONTROLDETAILS);
	mixerControlDetails.dwControlID = CurrParameter.dwControlID;
	mixerControlDetails.cChannels = cChannels;
	mixerControlDetails.cMultipleItems = CurrParameter.cMultipleItems;
	mixerControlDetails.cbDetails = sizeof(MIXERCONTROLDETAILS_UNSIGNED);
	mixerControlDetails.paDetails = &pmaci_fader->pmxcd_u[0];

	if ((err = mixerGetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_GETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u getting info on parameter ID #%u!", err, CurrParameter.dwControlID);
		return(err);
	}

	// Set the audio parameter to the value of the scroll bar
	pmaci_fader->pmxcd_u[uIndex].dwValue  = (DWORD)MulDiv((nRange - nValue), 0xFFFF, nRange);

	if ((err = mixerSetControlDetails((HMIXEROBJ)MixerHandle, &mixerControlDetails, MIXER_SETCONTROLDETAILSF_VALUE)))
	{
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, "Error #%u setting value of parameter ID #%u!", err, CurrParameter.dwControlID);
		return(err);
	}

	// NOTE: We do not need to set the scrollbar position here. Because we have told the
	// mixer device to inform us of any changes to a audio control's value by sending 
	// our main window an MM_MIXM_CONTROL_CHANGE message, that message will be sent to
	// us now that we've set the parameter's value above. When that happens, the main
	// window's procedure will send an MM_MIXM_CONTROL_CHANGE message to the "Parameter
	// Edit" dialog, which will in turn call changeFaderParameter() which actually sets
	// the scrollbar position

	return(0);
}





// ************************** initFaderDlg() **************************
// Called by faderDlgProc()'s WM_INITDIALOG to initialize the "Parameter
// Edit" (IDD_PARAMEDIT) dialog prior to displaying it for the first
// time.
//
// NOTE: The global 'CurrParameter' MIXERCONTROL struct must already be
// filled in with the current parameter, and 'CurrMixerLine' MIXERLINE
// struct must be filled in with its line.
//
// ARGS:
//		HWND hwnd:					Handle to "Parameter Edit" dialog.
//
//		LPMACONTROLINSTANCE pmaci:	Pointer to MACONTROLINSTANCE struct
//									filled in with the values of the
//									parameter being displayed.
//
// RETURNS:
//		0 if success, or non-zero if an error (-4 if not a type of fader
//		understood by this app, -1 if memory error, -5 if a window
//		can't be created, or other error # from the Mixer API).

MMRESULT initFaderDlg(HWND hwnd)
{
	#define FSB_DEF_STYLE   (WS_VISIBLE | WS_CHILD | SBS_VERT | WS_TABSTOP)

	static const TCHAR			szScrollBar[] = TEXT("scrollbar");

	PMACONTROLINSTANCE_FADER	pmaci_fader;
	UINT						u, v;
	HWND						hwndChild;
	RECT						rc;
	int							cxvsb;
	UINT						cChannels, cMultipleItems;
	UINT						uIndex;
	int							nRange;
	int							nPageInc;

	// Check that it's a legitimate parameter type for the fader class
	switch (CurrParameter.dwControlType)
	{
		case MIXERCONTROL_CONTROLTYPE_FADER:
		case MIXERCONTROL_CONTROLTYPE_VOLUME:
		case MIXERCONTROL_CONTROLTYPE_BASS:
		case MIXERCONTROL_CONTROLTYPE_TREBLE:
		case MIXERCONTROL_CONTROLTYPE_EQUALIZER:
			break;

		default:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Invalid fader type: 0x%.08X!", CurrParameter.dwControlType);
			return((MMRESULT)-4);
		}
	}

	// Set the dialog's title to the name of the parameter being edited
	SetWindowText(hwnd, &CurrParameter.szName[0]);

	// Display the minimum and maximum limits
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_BOUNDS);
	doWindowText(hwndChild, "dwMinimum=%lu, dwMaximum=%lu", CurrParameter.Bounds.dwMinimum, CurrParameter.Bounds.dwMaximum);

	// Display the steps
	hwndChild = GetDlgItem(hwnd, IDD_MACONTROL_TXT_METRICS);
	doWindowText(hwndChild, "cSteps=%lu", CurrParameter.Metrics.cSteps);

	// Determine how many arrays we'll need, based upon how many items there are,
	// and how many channels are in each item

	// Assume not mono
	cChannels = (UINT)CurrMixerLine.cChannels;

	// If a uniform parameter, then we have only a mono adjustment
	if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
		cChannels = 1;

	// Assume only 1 item
	cMultipleItems = 1;

	// If multiple items, then get how many items
	if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
		cMultipleItems = (UINT)CurrParameter.cMultipleItems;

	// Get a MACONTROLINSTANCE_FADER struct with an array of MIXERCONTROLDETAILS_UNSIGNED
	// structs for this parameter. The number of array structs are cMultipleItems * cChannels
	// so that we can set/retrieve all individual items on all channels simultaneously
	v = sizeof(MACONTROLINSTANCE_FADER) + (cChannels * cMultipleItems * sizeof(MIXERCONTROLDETAILS_UNSIGNED));
	if (!(pmaci_fader = (MACONTROLINSTANCE_FADER *)LocalAlloc(LPTR, v)))
 	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't get memory for sliders!");
 		return((MMRESULT)-1);
 	}

	// Save the pointer in the dialog's DWL_USER field so that other functions can retrieve it
	SetWindowLong(hwnd, DWL_USER, (LPARAM)pmaci_fader);

	// Figure out by how many units the PAGEUP and PAGEDOWN keys move
	nRange = (int)min(32767, CurrParameter.Metrics.cSteps - 1);
	nPageInc = nRange / 10;
	if (0 == nPageInc) nPageInc = 1;

	// Save these values so that setFaderParameter() and changeFaderParameter() can access them
	pmaci_fader->nRange	= nRange;
	pmaci_fader->nPageInc = nPageInc;

	// Figure out how wide to make each fader based upon how many faders we need
	// to squeeze into the IDC_PARAMEDIT_MULTICHANNEL groupbox
	cxvsb = GetSystemMetrics(SM_CXVSCROLL);

	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_MULTICHANNEL);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.right = rc.left + cxvsb;

	// Create the faders

	// ================== Update the individual item faders =================

	// Start with a unique window ID
	uIndex = IDD_MACONTROL_MULTICHANNEL_BASE;

	// For each channel...
	for (u = 0; u < cChannels; u++)
	{
		// and each item on this channel...
		for (v = 0; v < cMultipleItems; v++)
		{
			// Create a graphical fader, and then increment the window ID #
			if (!(hwndChild = CreateWindow(szScrollBar, 0, FSB_DEF_STYLE,
								rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
								hwnd, (HMENU)(uIndex++),
								MyInstance, NULL)))
			{
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create fader!");
				return((MMRESULT)-5);
			}

			// Set the range
			SetScrollRange(hwndChild, SB_CTL, 0, nRange, FALSE);

			// Move over for next item's fader
			rc.left  += cxvsb + 4;
			rc.right += cxvsb + 4;
		}

		// Add separation between this fader and the next channel's fader
		rc.left  += cxvsb;
		rc.right += cxvsb;
	}

	// ======================= Update the master faders ======================

	// Figure out how wide to make each fader based upon how many faders we have
	// to squeeze into the IDC_PARAMEDIT_UNIFORM groupbox
	hwndChild = GetDlgItem(hwnd, IDC_PARAMEDIT_UNIFORM);
	GetWindowRect(hwndChild, &rc);

	InflateRect(&rc, -10, -20);
	ScreenToClient(hwnd, (LPPOINT)&rc.left);
	ScreenToClient(hwnd, (LPPOINT)&rc.right);

	rc.right = rc.left + cxvsb;

	// Create the sliders
	for (v = 0; v < cMultipleItems; v++)
	{
		if (!(hwndChild = CreateWindow(szScrollBar, 0, FSB_DEF_STYLE,
							rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
							hwnd, (HMENU)(IDD_MACONTROL_UNIFORM_BASE + v),
							MyInstance, NULL)))
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "Can't create fader!");
			return((MMRESULT)-5);
		}

		SetScrollRange(hwndChild, SB_CTL, 0, nRange, FALSE);

		rc.left  += cxvsb + 4;
		rc.right += cxvsb + 4;
	}

	// =========================================================================

	// Send MM_MIXM_LINE_CHANGE and MM_MIXM_CONTROL_CHANGE messages to
	// IDD_PARAMEDIT dialog so that it updates its window per the
	// current settings of the current line and audio parameter. (ie,
	// Just like its window procedure would do if the operating system
	// sent it these messages)
	SendMessage(hwnd, MM_MIXM_LINE_CHANGE, (WPARAM)MixerHandle, CurrMixerLine.dwLineID);
	SendMessage(hwnd, MM_MIXM_CONTROL_CHANGE, (WPARAM)MixerHandle, CurrParameter.dwControlID);

	// Success
	return(0);
}





// *************************** faderDlgProc() ****************************
// Dialog procedure for the "Parameter Edit" (IDD_PARAMEDIT) dialog when
// it is displaying the settings of a fader (MIXERCONTROL_CT_CLASS_FADER)
// type of audio parameter.
//
// This dialog is opened when the user clicks the "Value" button for a
// fader type of control in the main window's dialog's listbox full of
// parameter names.
//
//	ARGS:
//		HWND hwnd:		Handle to DLG_MIXAPP_CONTROL dialog.
//
//		UINT uMsg:		The message type.
//
//		WPARAM wParam:	Depends upon uMsg.
//
//		LPARAM lParam:	Depends upon uMsg.
//
//	RETURNS:
//		Typically, TRUE if the message is handled, or FALSE if not.

BOOL APIENTRY faderDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			MMRESULT	err;

			// Create some child control windows, such as faders or checkmarks, which
			// will display the values of the channel masters and individual channels
			// for this audio parameter
			if ((err = initFaderDlg(hwnd)))

			// If an error, close the dialog
			{
				EndDialog(hwnd, err);
			}

			// If success, save the handle of this dialog in a global so that we'll
			// know it's open in case the operating system tells us that this audio
			// parameter has been updated
			else
				ghdlgControl = hwnd;

			return(TRUE);
		}

		// The operating system is notifying us that some line of the mixer we
		// currently have open is changed. We may need to now fetch that line's
		// new settings and update our window (assuming that we're currently
		// displaying that line's settings)
		case MM_MIXM_LINE_CHANGE:
		{
			changeFaderLine(hwnd, (HMIXER)wParam, lParam);
			return(TRUE);
		}

		// The operating system is notifying us that some audio parameter of the
		// mixer we currently have open is changed. We may need to now fetch
		// that parameter's new settings and update our window (assuming that
		// we're currently displaying that parameter's settings)
		case MM_MIXM_CONTROL_CHANGE:
		{
			changeFaderParameter(hwnd, (HMIXER)wParam, lParam);
			return(TRUE);
		}

		// The user is moving one of the graphical scroll bars to adjust the
		// value of the parameter
		case WM_VSCROLL:
		{
			setFaderParameter(hwnd, (HWND)lParam, LOWORD(wParam), HIWORD(wParam));
			return(TRUE);
		}

		case WM_COMMAND:
		{
			// If OK button, or window close button, close the dialog
			// and free memory
			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				case IDCANCEL:
				{
					HLOCAL		hl;

					// Retrieve the MACONTROLINSTANCE_FADER struct that we allocated when
					// we created the Parameter Edit dialog. The pointer to this array was stored
					// in the Parameter Edit dialog's DWL_USER field
					hl = (HLOCAL)GetWindowLong(hwnd, DWL_USER);

					// Close the dialog with SUCCESS
					EndDialog(hwnd, 0);

					// Indicate that the "Parameter Edit" dialog is no longer open
					ghdlgControl = 0;

					// Free the array
					LocalFree(hl);

//					break;
				}
			}
//			break;
		}
	}

	return(FALSE);
}
