// mactrls.c
//
// Manages the main window's listbox of audio control names for the
// currently selected audio line of the currently open Mixer. (The user
// selects the desired audio line via the main window's tree full of audio
// line names). This info is displayed in a listbox, where each text line
// in the listbox contains info upon one control of the line.
//
// The main window also has an "Info" button to click, which will open up
// the "Parameter Info" dialog containing information about whichever
// audio control is highlighted in the listbox. This source file contains
// the code to manages the "Parameter Info" (IDD_PARAMINFO) dialog.
//
// The main window also has a "Value" button, which will open up the
// "Parameter Edit" dialog containing information about the settings (ie,
// current value, range, control type, etc) for whichever audio control is
// highlighted in the listbox. This source file contains a stub which calls
// code in the various source files for the various types of controls (ie,
// fader, meter, switch, list, etc).
//
// NOTE: I use the term "parameter" as synonymous with "audio control".
//==========================================================================;

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>

#include "resource.h"
#include "mixapp.h"



const TCHAR		gszParamErr[] = TEXT("Error #%u getting info upon parameters!");
const TCHAR		gszParamErr2[] = TEXT("Error #%u getting info upon parameter ID 0x%08X!");
const TCHAR		gszParamOff[] = TEXT("This audio parameter is currently disabled.");
const TCHAR		gszUnsupported[] = TEXT("Can't edit a parameter of type=%.08lXh!");





// ************************* redrawParameters() *************************
// Fills the main window's listbox control with the names of all of
// the parameters (audio controls) in the currently selected line.
// 'CurrMixerLine' must already be filled in with info about the
// currently selected line.
//
// RETURNS:
//		0 if success, -1 if memory error, or error return from one of the
//		Mixer APIs.

long redrawParameters(void)
{
	HWND *				hwndList;
	MIXERLINECONTROLS	mixerLineControls;
	MMRESULT			err;

	// Get listbox control
	if ((hwndList = GetDlgItem(MainWindow, IDC_MAIN_PARAMLIST)))
	{
		// Hide and clear the listbox
		SetWindowRedraw(hwndList, FALSE);
		ListBox_ResetContent(hwndList);

		// Does this line have any parameters?
		if ((mixerLineControls.cControls = CurrMixerLine.cControls))
		{
			// Get memory for enough MIXERCONTROL structs to hold info on all parameters
			if (!(mixerLineControls.pamxctrl = (PMIXERCONTROL)LocalAlloc(LPTR, sizeof(MIXERCONTROL) * mixerLineControls.cControls)))
			{
				SetWindowRedraw(hwndList, TRUE);
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, TEXT("Out of memory!"));
				return(-1);
			}

			// Disable "Info" and "Settings" buttons while we're filling the listbox
			EnableWindow(GetDlgItem(MainWindow, IDC_MAIN_PARAMINFO), FALSE);
			EnableWindow(GetDlgItem(MainWindow, IDC_MAIN_PARAMEDIT), FALSE);

			// Get info upon all parameters
			mixerLineControls.cbStruct = sizeof(MIXERLINECONTROLS);
			mixerLineControls.dwLineID = CurrMixerLine.dwLineID;
			mixerLineControls.cbmxctrl = sizeof(MIXERCONTROL);

			if ((err = mixerGetLineControls((HMIXEROBJ)MixerHandle, &mixerLineControls, MIXER_GETLINECONTROLSF_ALL)))
			{
				SetWindowRedraw(hwndList, TRUE);
				LocalFree((HLOCAL)mixerLineControls.pamxctrl);
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, gszParamErr, err);
				return(err);
			}

			// For each parameter...
			for (mixerLineControls.dwLineID = 0; mixerLineControls.dwLineID < mixerLineControls.cControls; mixerLineControls.dwLineID++)
			{
				// Add the parameter's name to the listbox
				mixerLineControls.dwControlID = ListBox_AddString(hwndList, (LPSTR)(mixerLineControls.pamxctrl[mixerLineControls.dwLineID].szName));

				// Save its ID as the ItemData
				ListBox_SetItemData(hwndList, mixerLineControls.dwControlID, mixerLineControls.pamxctrl[mixerLineControls.dwLineID].dwControlID);
			}

			// Highlight the first parameter by default
			ListBox_SetCurSel(hwndList, 0);

			// Enable "Info" and "Settings" buttons now that we're done initializing
			// the dialog
			EnableWindow(GetDlgItem(MainWindow, IDC_MAIN_PARAMINFO), TRUE);
			EnableWindow(GetDlgItem(MainWindow, IDC_MAIN_PARAMEDIT), TRUE);

			// Free memory allocated for MIXERCONTROLS
			LocalFree((HLOCAL)mixerLineControls.pamxctrl);
		}

		// Allow listbox to be redrawn
		SetWindowRedraw(hwndList, TRUE);
	}

	return(0);
}





// ************************ editParameter() **************************
// Presents a dialog to display/alter a particular audio parameter's
// (ie, control's) settings. Depending upon the type of parameter,
// this will open one of several types of dialogs.
//
// Called when user clicks on the "Value" button in the main window.
//
// ARGS:
//		DWORD dwControlID:	ID # of the desired audio parameter.
//
// RETURNS:
//		0 if success, or -2 if parameter is disabled, -3 if editing of
//		this parameter is not supported, or Mixer API error number if
//		an error.

long editParameter(DWORD dwControlID)
{
	MIXERLINECONTROLS	mixerLineControls;
	int					err;

	// Get some info on the requested audio parameter
	mixerLineControls.cbStruct = sizeof(MIXERLINECONTROLS);
	mixerLineControls.dwControlID = dwControlID;
	mixerLineControls.cControls = 1;
	mixerLineControls.cbmxctrl = sizeof(MIXERCONTROL);
	mixerLineControls.pamxctrl = &CurrParameter;

	if ((err = mixerGetLineControls((HMIXEROBJ)MixerHandle, &mixerLineControls, MIXER_GETLINECONTROLSF_ONEBYID)))
	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, gszParamErr2, err, dwControlID);
		return(err);
	}

	// Is the parameter disabled?
	if (MIXERCONTROL_CONTROLF_DISABLED & CurrParameter.fdwControl)
	{
		doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, gszParamOff);
		return(-2);	// Disabled
	}

	// Indicate that we have the "Parameter Edit" dialog open (in case
	// the Mixer device sends our main window a message to update, and
	// we need to update this dialog too)
	gfDisplayingControl	= TRUE;

	// Save the ID of the parameter whose info is being displayed
	gdwControlID = dwControlID;

	// Assume an error
	err = -3;

	// Depending upon the type (ie, class) of audio parameter, open a dialog
	// appropriate for displaying its settings
	switch (MIXERCONTROL_CT_CLASS_MASK & CurrParameter.dwControlType)
	{
		case MIXERCONTROL_CT_CLASS_CUSTOM:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "CUSTOM class of controls not done yet!");
			break;
		}

		case MIXERCONTROL_CT_CLASS_METER:
		{
			// Open the "Parameter Edit" dialog and have meterDlgProc() service its messages
			err = DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_PARAMEDIT), MainWindow, meterDlgProc, (LPARAM)&CurrMixerLine);
			break;
		}

		case MIXERCONTROL_CT_CLASS_SWITCH:
		{
			// Open the "Parameter Edit" dialog and have switchDlgProc() service its messages
			err = DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_PARAMEDIT), MainWindow, switchDlgProc, (LPARAM)&CurrMixerLine);
			break;
		}

		case MIXERCONTROL_CT_CLASS_NUMBER:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "NUMBER class of controls not done yet!");
			break;
		}

		case MIXERCONTROL_CT_CLASS_SLIDER:
		{	
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, "SLIDER class of controls not done yet!");
			break;
		}

		case MIXERCONTROL_CT_CLASS_FADER:
		{
			// Open the "Parameter Edit" dialog and have faderDlgProc() service its messages
			err = DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_PARAMEDIT), MainWindow, faderDlgProc, (LPARAM)&CurrMixerLine);
			break;
		}

		case MIXERCONTROL_CT_CLASS_LIST:
		{
			// Open the "Parameter Edit" dialog and have listDlgProc() service its messages
			err = DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_PARAMEDIT), MainWindow, listDlgProc, (LPARAM)(LPVOID)&CurrMixerLine);
			break;
		}

		default:
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, gszUnsupported, CurrParameter.dwControlType);
//			break;
		}
	}

	// No longer displaying the "Parameter Edit" dialog
	gfDisplayingControl = FALSE;

	return(err);
}





// ************************** getControlType() ***************************
// Returns a string that describes the type of audio parameter (control)
// for the specified parameter.
//
// Args:
//		MIXERCONTROL *mixerControl:	MIXERCONTROL struct filled-in with
//									info about the desired parameter.
//
//	RETURNS:
//		Pointer to a string containing a mnuemonic name for the parameter's
//		type, or 0 if the type is unknown to this app.

LPCTSTR getControlType(LPMIXERCONTROL mixerControl)
{
	switch (mixerControl->dwControlType)
	{
		case MIXERCONTROL_CONTROLTYPE_CUSTOM:
		{
			return("Custom");
		}

		case MIXERCONTROL_CONTROLTYPE_BOOLEANMETER:
		{
			return("Boolean Meter");
		}

		case MIXERCONTROL_CONTROLTYPE_SIGNEDMETER:
		{
			return("Signed Meter");
		}

		case MIXERCONTROL_CONTROLTYPE_PEAKMETER:
		{
			return("Peak Meter");
		}

		case MIXERCONTROL_CONTROLTYPE_UNSIGNEDMETER:
		{
			return("Unsigned Meter");
		}

		case MIXERCONTROL_CONTROLTYPE_BOOLEAN:
		{
			return("Boolean Switch");
		}

		case MIXERCONTROL_CONTROLTYPE_ONOFF:
		{
			return("On/Off Switch");
		}

		case MIXERCONTROL_CONTROLTYPE_MUTE:
		{
			return("Mute Switch");
		}

		case MIXERCONTROL_CONTROLTYPE_MONO:
		{
			return("Mono Switch");
		}

		case MIXERCONTROL_CONTROLTYPE_LOUDNESS:
		{
			return("Loudness Switch");
		}

		case MIXERCONTROL_CONTROLTYPE_STEREOENH:
		{
			return("Stereo Enh Switch");
		}

		case MIXERCONTROL_CONTROLTYPE_BUTTON:
		{
			return("Button");
		}

		case MIXERCONTROL_CONTROLTYPE_DECIBELS:
		{
			return("Decibels Number");
		}

		case MIXERCONTROL_CONTROLTYPE_SIGNED:
		{
			return("Signed Number");
		}

		case MIXERCONTROL_CONTROLTYPE_PERCENT:
		{
			return("Percent Number");
		}

		case MIXERCONTROL_CONTROLTYPE_UNSIGNED:
		{
			return("Unsigned Number");
		}

		case MIXERCONTROL_CONTROLTYPE_SLIDER:
		{
			return("Slider");
		}

		case MIXERCONTROL_CONTROLTYPE_PAN:
		{
			return("Pan Slider");
		}

		case MIXERCONTROL_CONTROLTYPE_QSOUNDPAN:
		{
			return("Q-Sound Pan Slider");
		}

		case MIXERCONTROL_CONTROLTYPE_FADER:
		{
			return("Fader");
		}

		case MIXERCONTROL_CONTROLTYPE_VOLUME:
		{
			return("Volume Fader");
		}

		case MIXERCONTROL_CONTROLTYPE_BASS:
		{
			return("Bass Fader");
		}

		case MIXERCONTROL_CONTROLTYPE_TREBLE:
		{
			return("Treble Fader");
		}

		case MIXERCONTROL_CONTROLTYPE_EQUALIZER:
		{
			return("Equalizer Fader");
		}

		case MIXERCONTROL_CONTROLTYPE_SINGLESELECT:
		{
			return("Single-Sel List");
		}

		case MIXERCONTROL_CONTROLTYPE_MUX:
		{
			return("MUX List");
		}

		case MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT:
		{
			return("Multi-Sel List");
		}

		case MIXERCONTROL_CONTROLTYPE_MIXER:
		{
			return("Mixer List");
		}

		case MIXERCONTROL_CONTROLTYPE_MICROTIME:
		{
			return("Microsecond Time");
		}

		case MIXERCONTROL_CONTROLTYPE_MILLITIME:
		{
			return("Millisecond Time");
		}
	}

	return(0);
}





// ************************** parameterInfoDlgProc() ************************
// Dialog procedure for the "Parameter Info" (IDD_PARAMINFO) dialog which
// displays info about a particular audio parameter (ie, control).
//
// ARGS:
//		HWND hwnd:
//
//		UINT uMsg:
//
//		WPARAM wParam:
//
//		LPARAM lParam:
//
// RETURNS:

BOOL APIENTRY parameterInfoDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			MIXERLINECONTROLS	mxlc;
			MMRESULT			err;

			// Get info about this control
			mxlc.cbStruct = sizeof(MIXERLINECONTROLS);
			mxlc.dwControlID = lParam;
			mxlc.cControls = 1;
			mxlc.cbmxctrl = sizeof(MIXERCONTROL);
			mxlc.pamxctrl = &CurrParameter;

			if ((err = mixerGetLineControls((HMIXEROBJ)MixerHandle, &mxlc, MIXER_GETLINECONTROLSF_ONEBYID)))
            {
				doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, gszParamErr2, err, mxlc.dwControlID);
				break;
			}

            // Fill in the edit control with info about this mixer control
			else
			{
				HWND	hedit;

				// Get the multi-line edit control that fills this dialog
				hedit = GetDlgItem(hwnd, IDC_PARAMINFO_INFO);
				SetWindowFont(hedit, GetStockFont(SYSTEM_FIXED_FONT), FALSE);

                // Clear out edit control
				MEditPrintF(hedit, NULL);

				// Display ID # of the line that contains this mixer control
				MEditPrintF(hedit, "%25s: %.08lXh", (LPCTSTR)"Line ID", mxlc.dwLineID);

				// Display the ID # of this control
				MEditPrintF(hedit, "%25s: %.08lXh", (LPCTSTR)"Control ID", CurrParameter.dwControlID);

				// Display this control's type
                MEditPrintF(hedit, "~%25s: ", (LPCTSTR)"Control Type");
				switch(CurrParameter.dwControlType)
				{
					case MIXERCONTROL_CONTROLTYPE_CUSTOM:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_CUSTOM"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_BASS:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_BASS"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_EQUALIZER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_EQUALIZER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_FADER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_FADER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_TREBLE:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_TREBLE"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_VOLUME:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_VOLUME"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MIXER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MIXER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MULTIPLESELECT"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MUX:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MUX"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_SINGLESELECT:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_SINGLESELECT"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_BOOLEANMETER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_BOOLEANMETER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_PEAKMETER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_PEAKMETER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_SIGNEDMETER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_SIGNEDMETER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_UNSIGNEDMETER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_UNSIGNEDMETER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_DECIBELS:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_DECIBELS"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_PERCENT:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_PERCENT"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_SIGNED:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_SIGNED"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_UNSIGNED:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_UNSIGNED"));
						break;
					}
	
					case MIXERCONTROL_CONTROLTYPE_PAN:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_PAN"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_QSOUNDPAN:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_QSOUNDPAN"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_SLIDER:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_SLIDER"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_BOOLEAN:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_BOOLEAN"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_BUTTON:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_BUTTON"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_LOUDNESS:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_LOUDNESS"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MONO:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MONO"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MUTE:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MUTE"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_ONOFF:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_ONOFF"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_STEREOENH:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_STEREOENH"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MICROTIME:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MICROTIME"));
						break;
					}

					case MIXERCONTROL_CONTROLTYPE_MILLITIME:
					{
						MEditPrintF(hedit, "%s", TEXT("MIXERCONTROL_CONTROLTYPE_MILLITIME"));
						break;
					}
				}

				// Display its flag bits
				MEditPrintF(hedit, "~%25s: %.08lXh", (LPCTSTR)"Control Flags", CurrParameter.fdwControl);
				if (MIXERCONTROL_CONTROLF_DISABLED & CurrParameter.fdwControl)
					MEditPrintF(hedit, "~, disabled");
				if (MIXERCONTROL_CONTROLF_MULTIPLE & CurrParameter.fdwControl)
					MEditPrintF(hedit, "~, multiple");
				if (MIXERCONTROL_CONTROLF_UNIFORM & CurrParameter.fdwControl)
					MEditPrintF(hedit, "~, uniform");
				if (~(MIXERCONTROL_CONTROLF_UNIFORM | MIXERCONTROL_CONTROLF_MULTIPLE | MIXERCONTROL_CONTROLF_DISABLED) & CurrParameter.fdwControl)
					MEditPrintF(hedit, "~, *INVALID FLAGS*");
				MEditPrintF(hedit, "");

				// Display how many multiple items it has
				MEditPrintF(hedit, "%25s: %lu", (LPCTSTR)"Multiple Items", CurrParameter.cMultipleItems);

				// Display its short name
				MEditPrintF(hedit, "%25s: '%s'", (LPCTSTR)"Short Name", (LPTSTR)CurrParameter.szShortName);

				// Display its minimum/maximum limits, and its steps
				MEditPrintF(hedit, "%25s: %.08lXh", (LPCTSTR)"dwMinimum", CurrParameter.Bounds.dwMinimum);
				MEditPrintF(hedit, "%25s: %.08lXh", (LPCTSTR)"dwMaximum", CurrParameter.Bounds.dwMaximum);
				MEditPrintF(hedit, "%25s: %.08lXh", (LPCTSTR)"cSteps", CurrParameter.Metrics.cSteps);
			}

			return(TRUE);
		}

		case WM_COMMAND:
		{
			uMsg = GET_WM_COMMAND_ID(wParam, lParam);
			if ((IDOK == uMsg) || (IDCANCEL == uMsg))
			{
				EndDialog(hwnd, (IDOK == uMsg));
			}
//			break;
		}
    }

	return(FALSE);
}





// *********************** MixAppControlChange() *************************
// Called by our MainWindow dialog procedure when a Mixer device informs
// our app that some audio control of our currently open Mixer has
// changed (perhaps as a result of some other app changing some value).
// This gives us a chance to update our dialogs.
//
// ARGS:
//		HWND hwnd:				Window receiving the message from the Mixer.
//
//		HMIXER hmx:				Handle of Mixer whose control has changed.
//
//		DWORD dwControlID:		ID of the control being changed.
//
// RETURNS:

LRESULT MixAppControlChange(HWND hwnd, HMIXER hmx, DWORD dwControlID)
{
	// Are the Edit Parameter or Parameter Info dialogs being displayed right now?
	if (gfDisplayingControl &&

		// Is the parameter being changed the same as the one being displayed/edited
		// in the "Parameter Edit" or "Parameter Info" dialogs?
		(dwControlID == gdwControlID) &&

		ghdlgControl)
	{
		// Send a message to tell that dialog to update its display
		SendMessage(ghdlgControl, MM_MIXM_CONTROL_CHANGE, (WPARAM)hmx, dwControlID);
	}

	// We handled the message
	return(0);
}
