//==========================================================================
// mixapp.c
//
// This is an application that displays information about one of the installed
// Audio Mixer devices upon a system. A user can view information about the
// audio lines and audio controls on this mixer (such as their names and ID
// numbers), and can even alter the settings of various audio controls.
//
// This source file manages the main window (IDD_MAINWINDOW) for the app. This
// window uses a Tree graphical control (on the left side of the window) to
// display the types and names of all of the audio lines in the currently open
// Audio Mixer. (Only one Mixer can be open at a time). The name/type of each
// destination line is a root in that tree. The name/type of each source line
// is a leaf off of its respective destination line. One of the lines in the
// Tree can be highlighted (ie, selected) by the user. That line becomes the
// "currently selected line".
//
// More detailed information about the currently selected line is displayed in
// a readonly Edit box in the upper right side of the main window. (When
// a different audio line is selected in the Tree, this Edit box is updated to
// display information about that line).
//
// The names of all of the audio controls (ie, parameters) in the currently
// selected line are displayed in a listbox on the lower right side of the
// main window. (When a different audio line is selected in the Tree, this
// listbox is updated to reflect the audio controls in that line). The user
// can select one of these audio controls. That control becomes the "currently
// selected parameter".
//
// There are two buttons below the listbox labeled "Info" and "Value". These
// pertain to the currently selected parameter. When the user clicks on the
// Info button, a (IDD_PARAMINFO) dialog opens, displaying more detailed
// information about the currently selected parameter. When the user clicks
// on the Value button, a (IDD_PARAMEDIT) dialog opens displaying graphical
// controls that allow the user to change the parameter's value (ie, settings).
// The type of graphical controls presented in this new dialog depends upon
// the control type of the currently selected parameter.
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
// PURPOSE.
//
// Copyright (C) 1993 - 1997  Microsoft Corporation.  All Rights Reserved.
// Modified 2001 by Jeff Glatt
//==========================================================================

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <mmsystem.h>
#include <stdarg.h>

#include "resource.h"
#include "mixapp.h"


// App's Instance handle
HINSTANCE		MyInstance;

// Handle of main window
HWND			MainWindow = 0;

// MIXERLINE struct for currently selected line
MIXERLINE		CurrMixerLine;

// MIXERCONTROL struct for currently selected parameter
MIXERCONTROL	CurrParameter;

// To alert us when we have a dialog open that may need updating
// if the Mixer sends us a MM_MIXM_LINE_CHANGE or
// MM_MIXM_CONTROL_CHANGE message
BOOL			gfDisplayingControl	= 0;
DWORD			gdwControlID;
HWND			ghdlgControl;

// Strings
const TCHAR		gszNull[]	= TEXT("");
const TCHAR		gszCRLF[]	= TEXT("\r\n");
const TCHAR		gszLineErr[] = TEXT("Error %u getting info on Component ID %u!");
const TCHAR		gszNone[]	= TEXT("There are no mixer devices installed. This application is useless.");





// ******************************* doMsgBox() *************************
// Formats/displays a string (using wsprintf()) in a standard message
// box. The resulting string may be no larger than
// APP_MAX_STRING_ERROR_CHARS (defined in MIXAPP.H).
//
// ARGS:
//		HWND hwnd:			Handle to parent window for message box.
//
//		UINT fuStyle:		Style flags for MessageBox().
//
//		PTSTR pszFormat:	Format string used for wsprintf().
//
// RETURNS:
//		The result of MessageBox() function.

int doMsgBox(HWND hwnd, UINT fuStyle, LPCTSTR pszFormat, ...)
{
	va_list		va;
	TCHAR		buffer[APP_MAX_STRING_ERROR_CHARS];
	int			count;

    // Format the message
	va_start(va, pszFormat);
	count = wvsprintf(&buffer[0], pszFormat, va);
	va_end(va);

	// Check for an error
	if (count < lstrlen(pszFormat))
	{
		lstrcpy(&buffer[0], TEXT("Can't load error message!"));
		count = lstrlen(&buffer[0]);
	}

	// Set the messagebox's title the same as the main window. Use whatever remaining
	// space is in the error message buffer to copy the title
	if ((count + 1) < APP_MAX_STRING_ERROR_CHARS)
	{
		GetWindowText(MainWindow, &buffer[count + 1], APP_MAX_STRING_ERROR_CHARS - (count + 1));
	}
	else
	{
		// Just use a blank title
		count--;
	}

	// Display the messagebox and return its result
	return(MessageBox(hwnd, &buffer[0], &buffer[count + 1], fuStyle));
}






// ************************** doWindowText() *************************
// Formats a string and sets the specified window text to the result.
// The resulting string may be no larger than
// APP_MAX_STRING_ERROR_CHARS (defined in MIXAPP.H).
//
// ARGS:
//		HWND hwnd:			Handle to window to receive the new text.
//
//		PTSTR pszFormat:	Pointer to any valid format for wsprintf.
//
// RETURNS:
//		The number of bytes in the resulting window text.

int doWindowText(HWND hwnd, LPCTSTR pszFormat, ...)
{
	va_list		va;
	TCHAR		buffer[APP_MAX_STRING_ERROR_CHARS];
	int			n;

	// Format and display the string in the window...
	va_start(va, pszFormat);
	n = wvsprintf(&buffer[0], pszFormat, va);
	va_end(va);

	SetWindowText(hwnd, &buffer[0]);

	return(n);
}





// **************************** MEditPrintF() *************************
// Prints formatted text using a Multi-line Edit Control as if it
// were a standard console display. This is a very easy way to
// display small amounts of text information that can be scrolled
// and copied to the clip-board.
//
// ARGS:
//		HWND hedit:			Handle to a Multiline Edit control.
//
//		PTSTR pszFormat:	Pointer to any valid format for wsprintf. If
//							this argument is NULL, then the Multiline
//							Edit Control is cleared of all text.
//
//	RETURNS:
//		The number of characters written into the edit control.
//
//	Notes:
//		The pszFormat string can contain combinations of escapes that
//		modify the default behaviour of this function. Escapes are single
//		character codes placed at the _beginning_ of the format string.
//
//		Current escapes defined are:
//
//		~	:	Suppresses the default CR/LF added to the end of the
//				printed line. Since the most common use of this function
//				is to output a whole line of text with a CR/LF, that is
//				the default.

int MEditPrintF(HWND hedit, LPCTSTR pszFormat, ...)
{
	va_list		va;
	TCHAR		buffer[APP_MAX_STRING_RC_CHARS];
	int			n;
	BOOL        fCRLF;

	// If the pszFormat argument is NULL, then just clear all text in the edit control..
	if (!pszFormat)
	{
		SetWindowText(hedit, gszNull);
		return(0);
	}

	// Assume stuffing a newline into the end of the Edit box
	fCRLF = TRUE;

	// Search for escapes to modify default behaviour
	for (;;)
	{
		switch (*pszFormat)
		{
			case '~':
			{
				fCRLF = FALSE;
				pszFormat++;
				continue;
			}
		}

		break;
	}

	// Format the string
	va_start(va, pszFormat);
	n = wvsprintf(&buffer[0], pszFormat, va);
	va_end(va);

	// Insert the formatted text at the end of the Edit box
	Edit_SetSel(hedit, (WPARAM)-1, (LPARAM)-1);
	Edit_ReplaceSel(hedit, &buffer[0]);

	// Insert a newline into the Edit box?
	if (fCRLF)
	{
		Edit_SetSel(hedit, (WPARAM)-1, (LPARAM)-1);
		Edit_ReplaceSel(hedit, gszCRLF);
	}

	return(n);
}





// ***************************** MainWndProc() ***************************
// Main window procedure. This displays info upon all of the audio lines
// in the currently open Mixer device. The names of all lines are
// displayed in a Tree, where each text line in the Tree contains the name
// of one audio line of the Mixer. The destination lines are roots in the
// Tree, and the source lines are leaves off off their respective 
// destination lines. Whichever line is highlighted becomes the "currently
// selected line". Information about this line is displayed in a readonly
// edit control to the right of the Tree. Also, there is a listbox that
// lists all of the parameters (ie, audio controls) in that line.
//
// The main window also has some menus. A "Mixer devices..." menu item will
// open up a dialog listing all of the installed Mixer devices. From this
// second (IDD_MIXERS) dialog, a user can select which Mixer device he wants
// the main window to list lines of, and he can also query some information
// about that Mixer device. The main window also has some other menu items
// for changing the font used for display, and an "Refresh" item to refresh
// the window with the currently open Mixer's lines.
//
// ARGS:
//		Standard args for a dialog procedure.
//
// RETURNS:
//		Standard return for a dialog procedure.

long APIENTRY mainWndProc(HWND hwnd, UINT uMsg, UINT wParam, long lParam)
{
	switch(uMsg)
	{
		// #######################################
		// Here you would add cases to handle other
		// Windows messages, such as perhaps WM_PAINT
		// to draw text and graphics.
		// #######################################
		
		// ***************************************************************
		// ======================== Menu or Buttons ======================
		// Handles the WM_COMMAND message for the main window. This message
		// is sent to the window containing the menu item or control that
		// is being manipulated by the user.
		case WM_COMMAND:
		{
			switch (LOWORD(wParam))
			{
				// #######################################
				// Here you would add cases to handle other
				// menu items you may add to the main window.
				// Use a resource editor to load the MidSkele.rc
				// file and make it easy to edit/modify the
				// menus, accellerator tables, and dialog.
				// #######################################

				// "File -> Mixer driver,,," menu item
				case IDM_FILE_MIXER_DEVICE:
				{
					UINT		newID, origID;

					// Get the ID # of the currently opened mixer
					mixerGetID((HMIXEROBJ)MixerHandle, &origID, MIXER_OBJECTF_HMIXER);

					// Present the Mixers (IDD_MIXERS) dialog, passing it the ID # of
					// the currently opened mixer. Let that dialog do its thing, and
					// return the ID # of the mixer that the user wants to open
					newID = (UINT)DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_MIXERS), hwnd, (DLGPROC)mixerDevDlgProc, origID);

					// Is it the same mixer we had open before?
					if (newID != origID)
					{
						// No, so close the old mixer and open the new one, setting the latter
						// as the currently open mixer
						openMixer(hwnd, newID);

						// Refresh the main window's Tree of lines on this mixer
						redrawAudioLines();
					}

					break;
				}


				// "File -> Exit,,," menu item
				case IDM_FILE_EXIT:
				{
					FORWARD_WM_CLOSE(hwnd, SendMessage);
					break;
				}

				// "Update" menu item
				case IDM_UPDATE:
				{
					// Refresh the listing of audio lines
					redrawAudioLines();
					break;
				}

				// "Help -> About,,," menu item
				case IDM_FILE_ABOUT:
				{
					DialogBox(MyInstance, MAKEINTRESOURCE(IDD_ABOUTBOX), hwnd, (DLGPROC)aboutDlgProc);
					break;
				}

				// Parameter "Info" button
				case IDC_MAIN_PARAMINFO:
				{
					HWND		hwndCtls;
					int			n;

					// Get the listbox of parameters
					if ((hwndCtls = GetDlgItem(hwnd, IDC_MAIN_PARAMLIST)) &&

						// Determine upon which audio parameter the user selected
						(n  = ListBox_GetCurSel(hwndCtls)) != LB_ERR &&
							
						// Retrieve the ID of the parameter
						(n = SendMessage(hwndCtls, LB_GETITEMDATA, n, 0)) != LB_ERR)
					{

						// Display info about this control in the IDD_PARAMINFO dialog
						DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_PARAMINFO), hwnd, parameterInfoDlgProc, n);
					}
					break;
				}

				// Parameter "Value" button
				case IDC_MAIN_PARAMEDIT:
				{
					HWND		hwndCtls;
					int			n;

					// Get the listbox
					if ((hwndCtls = GetDlgItem(hwnd, IDC_MAIN_PARAMLIST)) &&

						// Determine upon which audio parameter the user selected
						(n  = ListBox_GetCurSel(hwndCtls)) != LB_ERR &&
							
						// Retrieve the ID of the parameter
						(n = SendMessage(hwndCtls, LB_GETITEMDATA, n, 0)) != LB_ERR)
					{
						// Display this control's current settings and allow user to adjust
						editParameter(n);
					}
					break;
				}

				// Listbox full of parameter names for the currently selected line
				case IDC_MAIN_PARAMLIST:
				{
					switch (GET_WM_COMMAND_CMD(wParam, lParam))
					{
						// Double-click in the listbox
						case LBN_DBLCLK:
						{
							HWND		hwndCtls;
							long		n;

							// Get the listbox
							if ((hwndCtls = GetDlgItem(hwnd, IDC_MAIN_PARAMLIST)) &&

								// Determine upon which audio parameter the user double-clicked
								(n  = ListBox_GetCurSel(hwndCtls)) != LB_ERR &&
							
								// Retrieve the ID of the parameter
								(n = SendMessage(hwndCtls, LB_GETITEMDATA, n, 0)) != LB_ERR)
							{
								// If holding down the CTRL key when he double-clicked, then bring
								// up the IDD_PARAMINFO dialog to show information about this
								// audio control
								if (GetKeyState(VK_CONTROL) < 0)
								{
									DialogBoxParam(MyInstance, MAKEINTRESOURCE(IDD_PARAMINFO), hwnd, parameterInfoDlgProc, n);
								}

								// Otherwise, display/edit this parameter's current settings
								else
								{
									editParameter(n);
								}
							}
						}
  					}
				}
			}

			break;
		}

		// ******************************************************************
		// ======================== Various Controls ========================
		// Handles the WM_NOTIFY message for the main window. This message
		// is sent to the window containing the control that is being
		// manipulated by the user.
		case WM_NOTIFY: 
		{
			// Tree control full of line names?
			if (wParam == IDC_MAIN_LINELIST)
			{
				switch(((LPNMHDR)lParam)->code)
				{
					// User is making a new selection
					case TVN_SELCHANGED: 
					{
						// Fetch the line ID # from the TV_ITEM's lParam field
						CurrMixerLine.dwLineID	= ((NM_TREEVIEW FAR *)lParam)->itemNew.lParam;

						// Get info on that line
						CurrMixerLine.cbStruct	= sizeof(MIXERLINE);
						if ((uMsg = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &CurrMixerLine, MIXER_GETLINEINFOF_LINEID)))
						{
							doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, gszLineErr, uMsg, CurrMixerLine.dwLineID);
							break;
						}

						// Display info about that line
						redrawLineInfo();

						// Fill in listbox with that line's Parameters
						redrawParameters();
					}
				}
			}

            break; 
		}

		// ******************************************************************
		// ======================== Background color ========================
		case WM_CTLCOLORSTATIC:
		{
			SetBkMode((HDC)wParam, TRANSPARENT);

			// Return the color that we want Windows to use when painting
			// the background of our static controls
			return((long)GetSysColorBrush(1));
		}

		case WM_CTLCOLORDLG:
		{
			// Return the color that we want Windows to use when painting
			// the background of our client window
			return((long)GetSysColorBrush(1));
		}

		// ******************************************************************
		// ====================== Create main window ========================
		case WM_INITDIALOG:
		{
			HICON	icon;

			// Load/set icon for System menu on the window. I put my icon
			// in this executable's resources. Note that windows frees this
			// when my window is closed
			if ((icon = LoadIcon(MyInstance, MAKEINTRESOURCE(IDI_MAIN_ICON))))
				SetClassLong(hwnd, GCL_HICON, (LONG)icon);

			// Save handle to main window in global since some other cases
			// here may reference it before we return from the CreateDialog()
			// call in WinMain()
			MainWindow = hwnd;
	
			return(1);
		}

		// ******************************************************************
		// Mixer Device is notifying us that some other app has changed an
		// audio line's state or parameter's setting

		case MM_MIXM_LINE_CHANGE:
		{
			return(MixAppLineChange(hwnd, (HMIXER)wParam, lParam));
		}

		case MM_MIXM_CONTROL_CHANGE:
		{
			return(MixAppControlChange(hwnd, (HMIXER)wParam, lParam));
		}

		// ******************************************************************
		// =================== User wants to close window ===================
		case WM_CLOSE:
		{
			// Close the main window (and terminate app)
			DestroyWindow(hwnd);

			return(TRUE);
		}

		case WM_DESTROY:
		{
 			// Post the WM_QUIT message to quit the message loop in WinMain()
			PostQuitMessage(0);

			return(TRUE);
		}
	}

	// Indicate that I didn't handle the msg
	return(FALSE);
}





// ***************************** WinMain() ****************************
// Program entry point, Called by the operating system as the initial
// entry point for a Windows application.
//
// ARGS:
//		HINSTANCE hinst:		Identifies the current instance of the
//								application.
//
//      HINSTANCE hinstPrev:	Identifies the previous instance of the
//								application (NULL if first instance).
//								For Win 32, this argument is always NULL.
//
//		LPSTR pszCmdLine:		Points to null-terminated unparsed command
//								line. This string is strictly ANSI
//								regardless of whether the application is
//								built for Unicode. To get the Unicode
//								equivalent, call the GetCommandLine()
//								function (Win 32 only).
//
//		int nCmdShow:			How the main window for the application is
//								to be shown by default.
//
// RETURNS:
//		The result from WM_QUIT message (in wParam of MSG structure) if
//		the program is able to enter its message loop. Returns 0 if
//		the program is not able to enter its message loop.

int PASCAL WinMain(HINSTANCE hinst, HINSTANCE hinstPrev, LPSTR pszCmdLine, int nCmdShow)
{
	MSG			msg;
	HACCEL		haccl;

	// Ensure that the common control DLL is loaded. 
	InitCommonControls();

	// Our documentation states that WinMain is supposed to return 0 if
	// we do not enter our message loop--so assume the worst...
	msg.wParam = 0;

	// Save our module handle in a global, since other functions may need it
	MyInstance = GetModuleHandle(0);

	//  WARNING DANGER WARNING DANGER WARNING DANGER WARNING DANGER WARNING
	//
	//		For Windows 3.X, BEFORE calling any other mixer API's, we must call
	//		the mixerGetNumDevs() function to let the mixer thunk library
	//		dynamically link to all the mixer API's! This is not needed for
	//		Win32.
	//
	//  WARNING DANGER WARNING DANGER WARNING DANGER WARNING DANGER WARNING
	if (!mixerGetNumDevs())
	{
		doMsgBox(NULL, MB_OK|MB_ICONEXCLAMATION, gszNone);
    	return(0);
	}

	// Create the main window. Its template is gotten from this EXE's resources
	if (!(MainWindow = CreateDialog(MyInstance, MAKEINTRESOURCE(IDD_MAINWINDOW), 0, mainWndProc)))
	{
		return(0);
	}

	// Open mixer device with an ID # of 0 by default. A real app would
	// allow configuration and set to previous selection. but this is not
	// a real app
	if (!openMixer(MainWindow, 0) || 

		// Display the names of all of this mixer's lines and the parameters in
		// the currently selected line (by default, the first line)
		redrawAudioLines())
	{
		// An error. Close the window and exit
		EndDialog(MainWindow, 0);
		return(0);
	}

	// Show the main window
	ShowWindow(MainWindow, nCmdShow);
	UpdateWindow(MainWindow);

	// Load accellerator table
	haccl = LoadAccelerators(MyInstance, MAKEINTRESOURCE(IDA_ACCELERATOR));

	// Get and handle messages
	while (GetMessage(&msg, NULL, 0, 0) == 1)
	{
		if (!IsDialogMessage(MainWindow, &msg))
		{
			if (!TranslateAccelerator(msg.hwnd, haccl, &msg))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
	}

	// Close any open mixer
	closeMixer();

	// Return result of WM_QUIT message
	return(msg.wParam);
}
