/*
 * Midi_Io -- A dynamic link library to handle opening/closing MIDI devices.
 *
 * The functions in this DLL callable by a program:
 *
 * OpenMidiIn()
 * CloseMidiIn()
 * PrintMidiInErrorMsg()
 * OpenMidiOut()
 * CloseMidiOut()
 * PrintMidiOutErrorMsg()
 * AllNotesOff()
 *
 * This must be linked with WINMM.LIB. For the release (ie, not debug) version, then you
 * should set your linker to ignore the default libraries. This will reduce code size.
 */

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>


/* ********************************** Globals ***************************** */
/* Data that each app gets a separate copy of */

/* Handle of MIDI In Device */
HMIDIIN			MidiInHandle = 0;

/* ID of MIDI In Device */
DWORD			MidiInID = 0;

/* Handle of MIDI Out Device */
HMIDIOUT		MidiOutHandle = 0;

/* ID of MIDI Out Device */
DWORD			MidiOutID = 0;




#if defined(_MSC_VER)
#pragma data_seg("Shared")
#endif

/* ============================== SHARED DATA ==============================
 * NOTE: I specify this data section to be Shared (ie, each program that uses
 * this shares these variables, rather than getting its own copies of these
 * variables). This is because, since I have only globals that are read-only
 * or whose value is the same for all processes, I don't need a separate copy
 * of these for each process that uses this DLL. In Visual C++'s Linker
 * settings, I add "/section:Shared,rws"
 */

/* For PrintMidiInErrorMsg() and PrintMidiOutErrorMsg() */
const char		StrangeErrMsg[] = "Strange error number returned!";
const char		InvPtrErrMsg[] = "Specified pointer is invalid!";
const char		MemLockErrMsg[] = "Unable to allocate/lock memory!";
const char		MidiOutErrorStr[] = "Midi Output -- Error";
const char		MidiInErrorStr[] = "Midi Input -- Error";

#if defined(_MSC_VER)
#pragma data_seg()
#endif







/*********************** PrintMidiInErrorMsg() **************************
 * Retrieves and displays an error message for the passed MIDI Out error
 * number. It does this using midiInGetErrorText() and MessageBox(). It
 * also prepends the name of the MIDI In device.
 *
 * window =	Handle to window that owns the error message box.
 * err =	Error number returned from a midiIn...() function call.
 ************************************************************************/

void WINAPI PrintMidiInErrorMsg(HWND window, unsigned long err)
{
	MIDIINCAPS		mic;
	char			buffer[MAXERRORLENGTH+16];
	char *			ptr;

	/* Copy the MIDI Input name to buffer[] */
	ptr = &buffer[0];
	if (!midiInGetDevCaps(MidiInID, &mic, sizeof(MIDIINCAPS)))
	{
		strcpy(&buffer[0], (const char *)&mic.szPname);
		ptr += strlen(&buffer[0]);
		*(ptr)++ = ':';
		*(ptr)++ = ' ';
	}

	if ((err = midiInGetErrorText(err, ptr, &buffer[MAXERRORLENGTH+15] - ptr)))
	{
		const char * msg;

		switch (err)
		{
			case MMSYSERR_BADERRNUM:
			{
				msg = &StrangeErrMsg[0];
				break;
			}
	
			case MMSYSERR_INVALPARAM:
			{
				msg = &InvPtrErrMsg[0];
				break;
			}

			default:
			{
				msg = &MemLockErrMsg[0];
			}
		}

		lstrcpy(ptr, msg);
	}

	MessageBox(window, ptr, &MidiInErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}





/******************************** CloseMidiIn() ********************************
 * Close MIDI In Device if it's open, and clear MidiInHandle.
 *
 * window =	Handle to window to contain any error message box.
 *
 * RETURNS: 0 if success, otherwise an error number. In the event of an error,
 * this displays a message box to the user.
 *******************************************************************************/

DWORD WINAPI CloseMidiIn(HWND window)
{
	DWORD	err;

	/* Is the device open? */
	if ((err = (DWORD)MidiInHandle))
	{
		/* Unqueue any buffers we added. If you don't
		 * input System Exclusive, you won't need this
		 */
		midiInReset(MidiInHandle);

		/* Close device */
		if (!(err = midiInClose(MidiInHandle)))
		{
			/* Clear handle so that it's safe to call CloseMidiIn() anytime */
			MidiInHandle = 0;
		}

		/* An error */
		else if (window)
		{
			/* Display the error */
			PrintMidiInErrorMsg(window, err);
		}
	}

	/* Return the error */
	return(err);
}





/* ******************************** OpenMidiIn() *********************************
 * Open MIDI In Device (MidiInID is the device ID). Stores handle in MidiInHandle.
 * Starts recording MIDI input. I use a MIDI callback here, but you can do
 * whatever you wish to do.
 *
 * window =	Handle to window to contain any error message box.
 * param =	Pointer to callback routine for MIDI Input.
 *
 * RETURNS: 0 if success, otherwise an error number. In the event of an error,
 * this displays a message box to the user.
 *********************************************************************************/

DWORD WINAPI OpenMidiIn(HWND window, DWORD param)
{
	DWORD	err;

	/* Is it not yet open? */
	if (!MidiInHandle)
	{
		/* Open MIDI Input and set Windows to call my
		 * callback function. You may prefer
		 * to have something other than CALLBACK_FUNCTION
		 */
		if (!(err = midiInOpen(&MidiInHandle, MidiInID, (DWORD)param, 0, CALLBACK_FUNCTION)))
		{
			/* Start recording Midi and return if SUCCESS */
			if (!(err = midiInStart(MidiInHandle)))
			{
				MIDIINCAPS		mic;

				/* Set the window title to the name of the opened MIDI Input */
				if (window && !midiInGetDevCaps(MidiInID, &mic, sizeof(MIDIINCAPS)))
				{
					SetWindowText(window, &mic.szPname[0]);
				}
				
				/* Return success */
				return(0);
			}
			
			/* ============== ERROR ============== */

			/* Close MIDI In and zero handle */
			CloseMidiIn(window);
		}

		/* Clear handle so that it's safe to call CloseMidiIn() */
		MidiInHandle = 0;

		/* Display the error */
		PrintMidiInErrorMsg(window, err);

		/* Return error */
		return((long)err);
	}

	/* Return success */
	return(0);
}







/*********************** PrintMidiOutErrorMsg() **************************
 * Retrieves and displays an error message for the passed MIDI Out error
 * number. It does this using midiOutGetErrorText() and MessageBox(). It
 * also prepends the name of the MIDI Out device.
 *
 * window =	Handle to window that owns the error message box.
 * err =	Error number returned from a midiIn...() function call.
 *************************************************************************/

void WINAPI PrintMidiOutErrorMsg(HWND window, unsigned long err)
{
	MIDIOUTCAPS		moc;
	char			buffer[MAXERRORLENGTH+16];
	char *			ptr;

	/* Copy the MIDI Output name to buffer[] */
	ptr = &buffer[0];
	if (!midiOutGetDevCaps(MidiOutID, &moc, sizeof(MIDIOUTCAPS)))
	{
		strcpy(&buffer[0], (const char *)&moc.szPname);
		ptr += strlen(&buffer[0]);
		*(ptr)++ = ':';
		*(ptr)++ = ' ';
	}

	/* Get the error message */
	if ((err = midiOutGetErrorText(err, ptr, &buffer[MAXERRORLENGTH+15] - ptr)))
	{
		const char * msg;

		switch (err)
		{
			case MMSYSERR_BADERRNUM:
			{
				msg = &StrangeErrMsg[0];
				break;
			}
	
			case MMSYSERR_INVALPARAM:
			{
				msg = &InvPtrErrMsg[0];
				break;
			}

			default:
			{
				msg = &MemLockErrMsg[0];
			}
		}

		lstrcpy(ptr, msg);
	}

	/* Display a message box in the MainWindow */
	MessageBox(window, ptr, &MidiOutErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}





/******************************** CloseMidiOut() *******************************
 * Close MIDI Out Device if it's open, and clear MidiOutHandle.
 *
 * window =	Handle to window that owns the error message box.
 *
 * RETURNS: 0 if success, otherwise an error number. In the event of an error,
 * this displays a message box to the user.
 *******************************************************************************/

DWORD WINAPI CloseMidiOut(HWND window)
{
	DWORD	err;

	/* Is the device open? */
	if ((err = (DWORD)MidiOutHandle))
	{
		/* If you have any system exclusive buffers that you sent
		 * via midiOutLongMsg(), and which are still being output,
		 * you may need to wait for their MIDIERR_STILLPLAYING flags
		 * to be cleared before you close the device. Some drivers
		 * won't close with pending output, and will give an error.
		 */

		/* Close device */
		if (!(err = midiOutClose(MidiOutHandle)))
		{
			/* Clear handle so that it's safe to call CloseMidiOut() anytime */
			MidiOutHandle = 0;
		}

		/* An error */
		else if (window)
		{
			/* Display the error */
			PrintMidiOutErrorMsg(window, err);
		}
	}

	/* Return the error */
	return(err);
}





/******************************** OpenMidiOut() *********************************
 * Open MIDI Out Device (MidiOutID is the device ID). Stores handle in MidiOutHandle.
 *
 * window =	Handle to window that owns the error message box.
 *
 * RETURNS: 0 if success, otherwise an error number. In the event of an error,
 * this displays a message box to the user.
 ********************************************************************************/

DWORD WINAPI OpenMidiOut(HWND window)
{
	DWORD	err;

	/* Is it not yet open? */
	if (!MidiOutHandle)
	{
		/* Open it */
		if ((err = midiOutOpen(&MidiOutHandle, MidiOutID, 0, 0, CALLBACK_NULL)))
		{
			/* ============== ERROR ============== */

			/* Display the error */
			PrintMidiOutErrorMsg(window, err);

			/* Clear handle so that it's safe to call CloseMidiOut() */
			MidiOutHandle = 0;

			/* Return error */
			return((long)err);
		}
	}

	/* Return success */
	return(0);
}





/*************************** AllNotesOff() ***************************
 * Turns all notes off on the 16 channels.
 *********************************************************************/

void WINAPI AllNotesOff(void)
{
	register unsigned char	i;

	if (MidiOutHandle)
	{
	 	/* Do 16 channels (starting with channel 15 since I count the first channel as 0) */
	 	i = 15;
	 	do
 		{
	 		/* Send an ALL NOTES OFF controller (and an ALL CONTROLLERS OFF
		     * too just to decidedly mute any notes being held by the
			 * sustain pedal controller)
			 */
			midiOutShortMsg(MidiOutHandle, 0x000079B0 | i);
			midiOutShortMsg(MidiOutHandle, 0x00007BB0 | i);
		} while(i--);
	}
}





/******************************** DllMain() ********************************
 * Automatically called by Win32 when the DLL is loaded or unloaded.
 ***************************************************************************/

#if defined(_MSC_VER)
#ifndef _DEBUG
BOOL WINAPI _DllMainCRTStartup(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
#else
BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)  /* <--- Doesn't replace startup code */
#endif
#else
BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
#endif
{
    switch(fdwReason)
	{
		/* ============================================================== */
		case DLL_PROCESS_ATTACH:
		{
			/*
			 * Here you would do any initialization that you want to do when
			 * the DLL loads. The OS calls this every time another program
			 * runs which uses this DLL. You should put some complementary
			 * cleanup code in the DLL_PROCESS_DETACH case.
			 */
			break;
		}

		/* ============================================================== */
		case DLL_THREAD_ATTACH:
		{
			/* We don't need to do any initialization for each thread of
			 * the program which uses this DLL, so disable thread messages.
			 */
			DisableThreadLibraryCalls(hinstDLL);
			break;
		}

/*
		case DLL_THREAD_DETACH:
		{
			break;
		}
*/
		/* ============================================================== */
		case DLL_PROCESS_DETACH:
		{
			/* If the MIDI IN or OUT handles are open, close them. Even if
			 * the program crashes, the default Win32 exception handler
			 * will call here, and so our handles will get freed
			 */
			CloseMidiIn(0);
			CloseMidiOut(0);
		}
	}

	/* Success */
	return(1);
}
