/* MidSkele -- An example of a minimal Windows C program that opens and uses MIDI devices.
 *
 * This file contains the WinMain entry point, as well as my main window procedure.
 *
 * This program doesn't do much. It's meant to serve as a skeleton upon which you
 * can quickly get up and running when building your own Windows MIDI programs in
 * C. It does the following:
 *
 * 1). Starts with WinMain(). Before opening the main window, I load any settings
 *     that the user had this program save the last time he ran it. Those settings
 *     were saved in the registry. Those settings include his choice of Input and
 *     Output Device ID numbers.
 * 2). The main window is opened. I use a dialog template that has been linked into
 *     this executable's resources. It's a lot easier to create such a template with
 *     Visual C++'s dialog editor, complete with menus and accellerator tables and
 *     an icon and any other controls, than it is to use CreateWindowEx and create
 *     all that stuff on the fly. I use CreateDialog which loads the dialog from the
 *     resource, and allows me to specify my own routine for handling messages sent
 *     to that dialog. The routine here is mainWndProc().
 * 3). I then enter my message loop in WinMain, processing messages as the user
 *     interacts with my window and its menus. But before I receive those WM_COMMAND
 *     messages, Windows will pass me my WM_ACTIVATEAPP message which is where I actually
 *     open and close the MIDI devices in order to allow sharing them with other
 *     programs (as explained in my tutorial at http://www.borg.com/~jglatt/tech/share.htm).
 *     I also set the title of the window to the opened MIDI Input Device.
 *
 *     One menu I have in this program is the MIDI menu which allows the user
 *     to pick a MIDI Input or Output from a listbox filled with the names of all Inputs
 *     or Outputs in his computer. It also features that "Hold Device" technique discussed
 *     in my article. I also have a file menu to allow him to save his settings. Furthermore,
 *     I have a help menu, referencing a "dummy" help book. You'll make your own help book.
 *
 *     An important note: The MIDI Input and Output handles, and the code to open/close
 *     them have been placed into a Dynamic Link Library called midi_io.dll. Why is this done?
 *     It is done because, when a WIN32 app crashes under Windows 2000/NT/XP, any MIDI handles
 *     opened by the app are not closed. (This is unlike other handles, which are cleaned up by
 *     WIN32's default exception handling). But this can be solved by putting code to close
 *     the MIDI handles in a DLL, as described in my article. The midi_io.dll must either be in
 *     the same directory as our MidSkele,exe, or in the Windows SYSTEM directory where the OS
 *     can find it.
 *
 * 4). The app really does nothing. It hangs around hogging the MIDI In and Out (but in an
 *     intelligent manner -- run two copies of this app and notice how they transparently
 *     share the same MIDI device, even with a single client driver), until the user closes
 *     the main window, and the app terminates. Of course, it cleans up after itself.
 */

/* Windows include files */
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>

/* Here, I include #define's for my symbols used in my RESOURCE (.RC) and SOURCE (.C) files */
#include "resource.h"

/* Here, I declare my global variables that are used by various SOURCE modules */
#include "globals.h"

/* Here is the include file for the MIDI routines that are in midi_io.c */
#include "midi_io.h"




/* *********************** GLOBALS ************************ */

/* Executable handle */
HINSTANCE		MyInstance;

/* My main window handle */
HWND			MainWindow = 0;

/* Accellerator table */
HACCEL			MainAccelTable;

/* Main window font */
HFONT			FontHandle;

/* Color # for main window background (This could be 0 to 6) */
unsigned long	MainWindowColor = 5;

/* A flags to implement MIDI device sharing */
unsigned char	NoRelinquishFlag = 0;

/* Error message */
const char			BadWindow[] = "Can't create window!";
const char			ErrorStr[] = "Initialization Error";

/* The name of my particular help book. Yours will be whatever you want */
const char			HelpFileName[] = "my_help.hlp";





/****************************** setMenuOptions() *****************************
 * Sets or clears the checkmark on the Hold Device menu item depending upon
 * NoRelinquishFlag's state.
 *****************************************************************************/

void setMenuOptions(HWND hwnd)
{
	HMENU			hmenu;
	unsigned long	flag;

	/* Get handle to menu */
	if ((hmenu = GetMenu(hwnd)))
	{
		/* Set "Hold Device" menu checkmark based upon NoRelinquishFlag */
		flag = MF_BYCOMMAND|MF_UNCHECKED;
		if (NoRelinquishFlag)
		{
			flag = MF_BYCOMMAND|MF_CHECKED;
		}

		CheckMenuItem(hmenu, IDM_HOLD, flag);
	}
}





/******************************* mainWndProc() *********************************
 * This is my message procedure for MainWindow. It is called by Windows whenever
 * there is a message for my window to process.
 *******************************************************************************/

long APIENTRY mainWndProc(HWND hwnd, UINT uMsg, UINT wParam, long lParam)
{
	switch(uMsg)
	{

		/* #######################################
		 * Here you would add cases to handle other
		 * Windows messages, such as perhaps WM_PAINT
		 * to draw text and graphics.
		 * #######################################
		 */
		
		
		/* ***************************************************************
		   ======================== Menu or Buttons ====================== */
		case WM_COMMAND:
		{
			switch (wParam)
			{

				/* #######################################
				 * Here you would add cases to handle other
				 * menu items you may add to the main window.
				 * Use a resource editor to load the MidSkele.rc
				 * file and make it easy to edit/modify the
				 * menus, accellerator tables, and dialog.
				 * #######################################
				 */
				
				/* -------------------- MENU: File -> Save setting */
		        case IDM_FILE_SAVEENV:
				{
					/* Save settings to the registry */
					saveSettings();

					return(0);
				}

				/* -------------------- MENU: MIDI -> Input Device */
		        case IDM_MIDI_INDEV:
				{
					/* Get user's choice of MIDI In and open it */
					doMidiInDevDlg();

					return(0);
				}

				/* -------------------- MENU: MIDI -> Output Device */
		        case IDM_MIDI_OUTDEV:
				{
					/* Get user's choice of MIDI Out and open it */
					doMidiOutDevDlg();

					return(0);
				}

				/* -------------------- MENU: MIDI -> Hold devices */
		        case IDM_HOLD:
				{
					/* Toggle  NORELINQUISHFLAG */
					NoRelinquishFlag ^= 1;

					/* Set menu checkmark per new flag value */
					setMenuOptions(hwnd);

					return(0);
				}

				/* -------------------- MENU: Help -> Contents */
		        case IDM_HELP:
				{
					/* Bring up the Contents page of the help book */
dohelp:				uMsg = HELP_FINDER;
dohelp2:			WinHelp(MainWindow, &HelpFileName[0], uMsg, 0);

  					return(0);
				}

				/* -------------------- MENU: Help -> Index */
		        case IDM_HELP_INDEX:
				{
					/* Bring up the Index page */
					uMsg = HELP_INDEX;
					goto dohelp2;
				}
			}

			break;
		}

		/* ****************************************************************** */
		/* ======================== Background color ======================== */
		case WM_CTLCOLORDLG:
		{
			/* Return the color that we want Windows to use when painting
			 * the background of our client window.
			 */
			return((long)GetSysColorBrush(MainWindowColor));
		}

		/* ****************************************************************** */
		/* ===================== User pressed F1 key ======================== */
		case WM_HELP:
		{
			/* Bring up the contents page of my Help book */
			goto dohelp;
		}

		/* ****************************************************************** */
		/* ========================= Font handling ========================== */
		case WM_SETFONT:
		{
			/* Store font for later use. We could use this in the WM_PAINT
			 * processing to output text in the same font as is used by
			 * Windows to display the titlebar and controls in the window.
			 */
			FontHandle = (HFONT)wParam;

			break;
		}

		/* ****************************************************************** */
		/* ====================== Create main window ======================== */
		case WM_INITDIALOG:
		{
			HICON	icon;

			/* Load/set icon for System menu on the window. I put my icon
			 * in this executable's resources. Note that windows frees this
			 * when my window is closed.
			 */
			if ((icon = LoadIcon(MyInstance,MAKEINTRESOURCE(IDI_MAIN_ICON))))
				SetClassLong(hwnd, GCL_HICON, (LONG)icon);
  
			/* Set menu checkmark for NoRelinquishFlag. We have already done
			 * the call to loadSettings(). Now we just need to update the
			 * menu.
			 */
			setMenuOptions(hwnd);

			return(1);
		}

		/* ****************************************************************** */
		/* =============== App is gaining or losing activation ============== */
		case WM_ACTIVATEAPP:
		{
			/* Is app losing activation? */
			if (!wParam)
			{
				/* Does user want the feature where this app relinquishes its
				 * access to MIDI In and Out when user switches to another app?
				 */
				if (!NoRelinquishFlag)
				{
					/* Close the MIDI devices (so other apps can use them) */
					if (!CloseMidiIn(hwnd))

						/* Set the window title to indicate that MIDI devices are closed */
						SetWindowText(hwnd, "No MIDI devices open");
					
					CloseMidiOut(hwnd);

				
				}
			}

			/* App gaining activation */
			else
			{
				/* Post a message to myself to reopen MIDI devices
				 * later, if they're not already open 
				 */
				PostMessage(hwnd, WM_USER, (WPARAM)0, (LPARAM)0);
			}
	
			/* Allow windows to continue handling WM_ACTIVATEAPP */
			break;
		}

		/* ****************************************************************** */
		/* I sent a WM_USER message to myself during WM_ACTIVATEAPP
         * to open the MIDI In and Out devices.
		 */
		case WM_USER:
		{
			/* Open MIDI In */
			OpenMidiIn(hwnd, (DWORD)midiInputEvt);

			/* Open MIDI Out */
			OpenMidiOut(hwnd);

			/* Handled this message */
			return(1);
		}

		/* ****************************************************************** */
		/* =================== User wants to close window =================== */
		case WM_CLOSE:
		{
			/* Stop any recording, close any MIDI Input */
			CloseMidiIn(hwnd);

			/* Close any Midi Output device */
			CloseMidiOut(hwnd);

			/* Close this window */
			DestroyWindow(hwnd);

			return(1);
		}

		case WM_DESTROY:
		{
 			/* Post the WM_QUIT message to quit the message loop in WinMain() */
			PostQuitMessage(0);

			return(1);
		}
	}

	/* Indicate that I didn't handle the msg */
	return(0);
} 





/* ****************************** WinMain() ******************************** */
/* * Program Entry point */
/* ************************************************************************* */

int WINAPI WinMain(HINSTANCE hinstExe, HINSTANCE hinstPrev, LPSTR lpszCmdLine, int nCmdShow)
{
	MSG		msg;

	/* Save Instance handle which I need when opening the MIDI Input/Output Dialog
	 * box. That dialog is linked into this executable's resources, so I need to pass
	 * this handle to DialogBox() when I open that dialog
	 */
	MyInstance = hinstExe;

	/* Loads Windows common control's DLL  */
	InitCommonControls();

	/* #######################################
	 * Here you may do other initialization
	 * that happens before the main window
	 * opens. Remember that MIDI In/Out devices
	 * are not open yet.
	 *
	 * One thing that you may choose to do is
	 * load some settings that the user saved
	 * previously, such as the device IDs for
	 * the MIDI In and Out devices. A good place
	 * to save those is in the Windows registry.
	 * In fact, that is what I'll do right now.
	 * ####################################### */

	/* Load settings from my registry key */
	loadSettings();

	/* Create Main window. The WM_ACTIVATEAPP will post a message to open MIDI input and output */
	if (!(MainWindow = CreateDialog(MyInstance, MAKEINTRESOURCE(IDD_MAINWINDOW), 0, mainWndProc)))
	{
		CloseMidiIn(0);
		MessageBox(0, &BadWindow[0], &ErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
		return(-1);
	}

	/* Load Accellerator Table */
	MainAccelTable = LoadAccelerators(MyInstance, MAKEINTRESOURCE(IDA_ACCELERATOR));

	/* Show the window with default size/position */
	ShowWindow(MainWindow, SW_SHOWDEFAULT);
	UpdateWindow(MainWindow);

	/* Here's my message loop which is executed until the user closes
	 * down my main window.
	 */
	
	/* Get the next msg (until WM_QUIT) */
	while (GetMessage(&msg, 0, 0, 0) == 1)
	{
		/* Apply accelerator table */
		if (!TranslateAccelerator(msg.hwnd, MainAccelTable, &msg))
		{
			TranslateMessage(&msg);
		}

		/* Send msg to window procedure */
		DispatchMessage((CONST MSG *)&msg);
	}

	/* Free any help instance */
	WinHelp(MainWindow, &HelpFileName[0], HELP_QUIT, 0);

	/* Exit */
	return(0);
}
