/*
You must link with winmm.lib. If using Visual C++, go to Build->Settings. Flip to the Link page,
and add winmm.lib to the library/object modules.
*/

#include <windows.h>
#include <stdio.h>
#include <mmsystem.h>
#include <mmreg.h>		/* Defines the Manufacturer and Product IDs. This is not available for Win3.1 compilers */


const char  *Tech_IDs[] = {"MIDI output", "Internal Synth", "Square Wave Synth", "FM synth", "MIDI Mapper"};


// ************************ GetMIDIManfacturer() ***********************
// Returns a pointer to a string that names the Manufacturer associated
// with the passed id. Returns a pointer to a null string if the ID is
// undefined.

char Unknown[] = "<unknown>";

// MIDI Manufacturer IDs
char  *Man1_IDs[] = {"",
					"Microsoft",
					"Creative Labs",
					"Media Vision",
					"Fujitsu"};

char  *Man2_IDs[] = {"Artisoft",
					"Turtle Beach",
					"IBM",
					"Vocaltec",
					"Roland",
					"DSP Solutions",
					"NEC",
					"ATI",
					"Wang Labs",
					"Tandy",
					"Voyetra",
					"Antex",
					"ICL Personal Systems",
					"Intel",
					"Advanced Gravis",
					"Video Associates Labs",
					"InterActive Inc.",
					"Yamaha",
					"Everex",
					"Echo Speech",
					"Sierra Semiconductor",
					"Computer Aided Tech",
					"APPS Software",
					"DSP Group",
					"microEngineering",
					"Computer Friends",
					"ESS Tech",
					"Audio, Inc.",
					"Motorola",
					"Canopus",
					"Seiko Epson",
					"Truevision",
					"Aztech Labs",
					"Videologic",
					"SCALACS",
					"Korg",
					"Audio Processing Tech",
					"ICS",
					"Iterated Systems",
					"Metheus",
					"Logitech",
					"Winnov",
					"NCR",
					"EXAN",
					"AST Research",
					"Willow Pond",
					"Sonic Foundry",
					"Vitec",
					"MOSCOM",
					"Silicon Soft"
					"","","",
					"Supermac",
					"Audio Processing Tech",
					"",
					"Speech Compression",
					"",
					"Dolby Labs",
					"OKI",
					"AuraVision",
					"Olivetti",
					"I/O Magic",
					"Matsushita",
					"Control Resources",
					"Xebec",
					"New Media",
					"Natural MicroSystems",
					"Lyrrus",
					"Compusic",
					"OPTi Computers"
					"","",
					"Dialogic"};

char * GetMidiManufacturer(unsigned long id)
{
	if (id < 5)
	{
		return(Man1_IDs[id]);
	}

	// Return a null string for unknown IDs
	if (id < 20 || id > 93)
	{
		return(Man1_IDs[0]);
	}

	return(Man2_IDs[id - 20]);
}





// ************************** GetMIDIProduct() ************************
// Returns a pointer to a string that names the Product associated
// with the passed id. Returns a pointer to a null string if the ID is
// undefined.
//
// Note: I haven't included the product names for all of the defined
// Manufacturers. I only included those I know for Microsoft, Roland,
// IBM, Creative Labs, MediaVision, ESS, Aztech, Yamaha, and Opti.

typedef struct _PROD_ID
{
	unsigned long	Id;
	char *			Name;
} PROD_ID;

char MPUOut[] = "MPU401 compatible MIDI out";
char MPUIn[] = "MPU401 compatible MIDI in";
char SBOut[] = "SB MIDI Out";
char SBIn[] = "SB MIDI In";

// MS MIDI Product IDs
PROD_ID MS_ProdIDs[] = {MM_MIDI_MAPPER,"Midi Mapper",
					MM_SNDBLST_MIDIOUT,&SBOut[0],
					MM_SNDBLST_MIDIIN,&SBIn[0],
					MM_SNDBLST_SYNTH,"SB internal synth",
					MM_ADLIB,"Ad Lib Compatible synth",
					MM_MPU401_MIDIOUT, &MPUOut[0],
					MM_MPU401_MIDIIN,&MPUIn[0],
					MM_MSFT_WSS_FMSYNTH_STEREO,"Stereo FM synth",
					MM_MSFT_WSS_OEM_FMSYNTH_STEREO,"OEM Stereo FM synth",
					MM_MSFT_GENERIC_MIDIIN,"Generic MIDI In",
					MM_MSFT_GENERIC_MIDIOUT,"Generic MIDI Out",
					MM_MSFT_GENERIC_MIDISYNTH,"Generic MIDI synth",
					MM_WSS_SB16_MIDIIN,"SB16 midi in",
					MM_WSS_SB16_MIDIOUT,"SB16 midi out",
					MM_WSS_SB16_SYNTH,"SB16 FM synth",
					MM_WSS_SBPRO_MIDIIN,"SB Pro midi in",
					MM_WSS_SBPRO_MIDIOUT,"SB Pro midi out",
					MM_WSS_SBPRO_SYNTH,"SB Pro FM synth",
					MM_MSFT_WSS_NT_FMSYNTH_STEREO,"WSS NT FM synth",
					MM_MSFT_SB16_MIDIIN,"SB16 midi in",
					MM_MSFT_SB16_MIDIOUT,"SB16 midi out",
					MM_MSFT_SB16_SYNTH,"SB16 FM synth",
					MM_MSFT_SBPRO_MIDIIN,"SB Pro midi in",
					MM_MSFT_SBPRO_MIDIOUT,"SB Pro midi out",
					MM_MSFT_SBPRO_SYNTH,"SB Pro FM synth",
					MM_MSFT_MSOPL_SYNTH,"Yamaha OPL2/OPL3 FM synth",
					0, 0};

// CL Product IDs
PROD_ID CL_ProdIDs[] = {MM_CREATIVE_MIDIOUT,&SBOut[0],
					MM_CREATIVE_MIDIIN,&SBIn[0],
					MM_CREATIVE_FMSYNTH_MONO,"SB FM Mono synth",
					MM_CREATIVE_FMSYNTH_STEREO,"SB FM Stereo synth",
					MM_CREATIVE_MIDI_AWE32,"AWE Wavetable synth",
					MM_CREATIVE_AUX_MIDI,"Aux MIDI",
					0, 0};

// MediaVision Product IDs
PROD_ID MV_ProdIDs[] = {MM_PROAUD_MIDIOUT,"Pro Audio MIDI out",
					MM_PROAUD_MIDIIN,"Pro Audio MIDI in",
					MM_PROAUD_SYNTH, "Pro Audio FM synth",
					MM_THUNDER_SYNTH, "Thunder Board synth",
					MM_TPORT_SYNTH, "Audio Port synth",
					MM_PROAUD_PLUS_MIDIOUT,"PAS+ MIDI out",
					MM_PROAUD_PLUS_MIDIIN,"PAS+ MIDI in",
					MM_PROAUD_PLUS_SYNTH, "PAS+ synth",
					MM_PROAUD_16_MIDIOUT,"PAS16 MIDI out",
					MM_PROAUD_16_MIDIIN,"PAS16 MIDI in",
					MM_PROAUD_16_SYNTH, "PAS16 synth",
					MM_STUDIO_16_MIDIOUT,"Studio 16 MIDI out",
					MM_STUDIO_16_MIDIIN,"Studio 16 MIDI in",
					MM_STUDIO_16_SYNTH, "Studio 16 synth",
					MM_CDPC_MIDIOUT,"CDPC MIDI out",
					MM_CDPC_MIDIIN,"CDPC MIDI in",
					MM_CDPC_SYNTH, "CDPC synth",
					MM_OPUS401_MIDIOUT,"Opus MV 1208 MIDI out",
					MM_OPUS401_MIDIIN,"Opus MV 1208 MIDI in",
					MM_OPUS1208_SYNTH, "Opus MV 1208 synth",
					MM_OPUS1216_MIDIOUT,"Opus MV 1216 MIDI out",
					MM_OPUS1216_MIDIIN,"Opus MV 1216 MIDI in",
					MM_OPUS1216_SYNTH, "Opus MV 1216 synth",
					0, 0};

// IBM Product IDs
PROD_ID IBM_ProdIDs[] = {MM_IBM_PCMCIA_SYNTH, "IBM synth",
					MM_IBM_PCMCIA_MIDIIN,"IBM MIDI in",
					MM_IBM_PCMCIA_MIDIOUT,"IBM MIDI out",
					0, 0};

// Roland
PROD_ID Rol_ProdIDs[] = {MM_MPU401_MIDIOUT, &MPUOut[0],
					MM_MPU401_MIDIIN,&MPUIn[0],
					MM_ROLAND_MPU401_MIDIOUT,"MPU401 MIDI out",
					MM_ROLAND_MPU401_MIDIIN,"MPU401 MIDI in",
					MM_ROLAND_SMPU_MIDIOUTA, "SuperMPU MIDI Out 1",
					MM_ROLAND_SMPU_MIDIOUTB, "SuperMPU MIDI Out 2",
					MM_ROLAND_SMPU_MIDIINA, "SuperMPU MIDI In 1",
					MM_ROLAND_SMPU_MIDIINB, "SuperMPU MIDI In 2",
					MM_ROLAND_SC7_MIDIOUT, "SC-7 MIDI Out",
					MM_ROLAND_SC7_MIDIIN, "SC-7 MIDI In",
					MM_ROLAND_SERIAL_MIDIOUT, "Serial MIDI Out",
					MM_ROLAND_SERIAL_MIDIIN, "Serial MIDI In",
					0, 0};

// Yamaha
PROD_ID Yam_ProdIDs[] = {MM_YAMAHA_GSS_SYNTH,"GS synth",
					MM_YAMAHA_GSS_MIDIOUT,"GS MIDI out",
					MM_YAMAHA_GSS_MIDIIN,"GS MIDI in",
					0, 0};

// ESS
PROD_ID ESS_ProdIDs[] = {MM_ESS_AMSYNTH,"AM synth",
					MM_ESS_AMMIDIOUT,"AM MIDI out",
					MM_ESS_AMMIDIIN,"AM MIDI in",
					MM_ESS_MPU401_MIDIOUT,"MPU401 MIDI out",
					MM_ESS_MPU401_MIDIIN,"MPU401 MIDI in",
					MM_EPS_FMSND,"FM Synth",
					0, 0};
// Aztech
PROD_ID Az_ProdIDs[] = {MM_AZTECH_MIDIOUT,"MIDI out",
					MM_AZTECH_MIDIIN,"MIDI in",
					MM_AZTECH_PRO16_FMSYNTH,"Pro16 FM synth",
					MM_AZTECH_DSP16_FMSYNTH,"Dsp16 FM synth",
					MM_AZTECH_DSP16_WAVESYNTH,"Dsp16 Wavetable synth",
					0, 0};

// OPTI
PROD_ID Op_ProdIDs[] = {MM_OPTI_M16_FMSYNTH_STEREO,"M16 FM synth",
					MM_OPTI_M16_MIDIIN,"M16 MIDI in",
					MM_OPTI_M16_MIDIOUT,"M16 MIDI out",
					MM_OPTI_P16_FMSYNTH_STEREO,"P16 FM synth",
					MM_OPTI_P16_MIDIIN,"P16 MIDI in",
					MM_OPTI_P16_MIDIOUT,"P16 MIDI out",
					MM_OPTI_M32_MIDIIN,"M32 MIDI in",
					MM_OPTI_M32_MIDIOUT,"M32 MIDI out",
					MM_OPTI_M32_SYNTH_STEREO,"M32 synth",
					0, 0};

char * GetMidiProduct(unsigned long id, unsigned long prod)
{
	PROD_ID * prod_id;

	switch(id)
	{
		case MM_MICROSOFT:
			prod_id = &MS_ProdIDs[0];		
			break;

		case MM_CREATIVE:
			prod_id = &CL_ProdIDs[0];		
			break;

		case MM_MEDIAVISION:
			prod_id = &MV_ProdIDs[0];		
			break;

		case MM_IBM:
			prod_id = &IBM_ProdIDs[0];		
			break;

		case MM_ROLAND:
			prod_id = &Rol_ProdIDs[0];		
			break;

		case MM_YAMAHA:
			prod_id = &Yam_ProdIDs[0];		
			break;

		case MM_ESS:
			prod_id = &ESS_ProdIDs[0];		
			break;

		case MM_AZTECH:
			prod_id = &Az_ProdIDs[0];		
			break;

		case MM_OPTI:
			prod_id = &Op_ProdIDs[0];		
			break;

		default:
			return(Man1_IDs[0]);
	}

	while (prod_id->Name)
	{
		if (prod_id->Id == prod) return(prod_id->Name);
		prod_id++;
	}

	return(Man1_IDs[0]);
}





/* ******************************** main() ******************************** */

int main(int argc, char **argv)
{
	MIDIOUTCAPS		moc;
	MIDIINCAPS		mic;
	long			iNumDevs, i;
	char *			ptr;
	char			buffer[80];
	unsigned char	chr;

	/* Get the number of MIDI Out devices in this computer */
	iNumDevs = midiOutGetNumDevs();

	/* Note that I'm going to start my count with a device ID of -1 (instead of 0). Why?
	   So that I also include MIDI Mapper, whose ID is -1, in the list if it is installed
	*/

	/* Go through all of those devices, displaying their names */
	for (i = -1; i < iNumDevs; i++)
	{
		if (i == -1) printf("MIDI Output Devices =======================\r\n\n");
		
		/* Get info about the next device */
		if (!midiOutGetDevCaps(i, &moc, sizeof(MIDIOUTCAPS)))
		{
			/* Display its Device ID and name */
			printf("\r\n    Device ID #%ld: %s\r\n", i, moc.szPname);

			/* Display its Manufacturer */
			ptr = GetMidiManufacturer(moc.wMid);
			if (!(*ptr))
			{
				ptr = &Unknown[0];
			}
			printf("Manufacturer: %s\r\n", (const char *)ptr);

			/* Display its Product name */
			ptr = GetMidiProduct(moc.wMid, moc.wPid);
			if (!(*ptr))
			{
				ptr = &Unknown[0];
			}
			printf("Product Name: %s\r\n", (const char *)ptr);

			/* Driver Version */
			printf("Driver Version: %u.%u\r\n", moc.vDriverVersion>>8, moc.vDriverVersion & 0x00FF);

			/* MultiTimbral Parts, and Polyphony */
			printf("MultiTimbral Parts: %u   Polyphony: %u\r\n", moc.wVoices, moc.wNotes);

			/* Channels */
			ptr = &buffer[0];
			for (chr=0; chr<16; chr++)
			{
				if ((0x0001<<chr) & moc.wChannelMask)
				{
					if (ptr != &buffer[0])
					{
						*(ptr)++ = ',';
						*(ptr)++ = ' ';
					}
					if (chr > 8)
					{
						*(ptr)++ = '1';
						*(ptr)++ = chr - 9 + 0x30;
					}
					else
					{
						*(ptr)++ = chr + 0x31;
					}
				}
			}
			*ptr = 0;
			printf("Channels: %s\r\n", &buffer[0]);

			/* Hardware Type */
			ptr = &Unknown[0];
			if (moc.wTechnology && moc.wTechnology <= MOD_MAPPER)
			{
				ptr = (char *)Tech_IDs[moc.wTechnology-1];
			}
			printf("Hardware type: %s\r\n", ptr);

			/* Additional Support */
			ptr = &buffer[0];
			if (moc.dwSupport & (MIDICAPS_VOLUME|MIDICAPS_LRVOLUME))
			{
				if (moc.dwSupport & MIDICAPS_LRVOLUME)
				{
					strcpy(&buffer[0], "Separate left-right volume");
				}
				else
				{
					strcpy(&buffer[0], "Volume");
				}

				ptr += strlen(&buffer[0]);
			}

			if (moc.dwSupport & MIDICAPS_CACHE)
			{
				if (ptr != &buffer[0])
				{
					*(ptr)++ = ',';
					*(ptr)++ = ' ';
				}
				strcpy(ptr, "Patch caching");
				ptr += 13;
			}

			if (moc.dwSupport & MIDICAPS_STREAM)
			{
				if (ptr != &buffer[0])
				{
					*(ptr)++ = ',';
					*(ptr)++ = ' ';
				}
				strcpy(ptr, "Driver does midiStreamOut()");
				ptr += 27;
			}
				
			if (ptr != &buffer[0])
			{
				*(ptr)++ = '.';
			}
			*ptr = 0;

			printf("Supports: %s\r\n", &buffer[0]);
		}
	}

	/* Get the number of MIDI In devices in this computer */
	iNumDevs = midiInGetNumDevs();

	/* Go through all of those devices, displaying their names */
	for (i = 0; i < iNumDevs; i++)
	{
		if (!i) printf("\r\nMIDI Input Devices =======================\r\n");
		
		/* Get info about the next device */
		if (!midiInGetDevCaps(i, &mic, sizeof(MIDIINCAPS)))
		{
			/* Display its Device ID and name */
			printf("\r\n    Device ID #%ld: %s\r\n", i, mic.szPname);
		
			/* Display its Manufacturer */
			ptr = GetMidiManufacturer(mic.wMid);
			if (!(*ptr))
			{
				ptr = &Unknown[0];
			}
			printf("Manufacturer: %s\r\n", (const char *)ptr);

			/* Display its Product name */
			ptr = GetMidiProduct(mic.wMid, mic.wPid);
			if (!(*ptr))
			{
				ptr = &Unknown[0];
			}
			printf("Product Name: %s\r\n", (const char *)ptr);

			/* Driver Version */
			printf("Driver Version: %u.%u\r\n", mic.vDriverVersion>>8, mic.vDriverVersion & 0x00FF);
		}
    }

	return(0);
}
