/* ============================== MIDIFILE.H =================================
 * Include file for Jeff Glatt's MIDIFILE.DLL. Contains definitions of various
 * structures/constants found within a MIDI file, and structures/constants used
 * by the MIDIFILE.DLL to read/write MIDI files.
 ========================================================================== */

#ifndef _INC_MIDIFILE
#define _INC_MIDIFILE

#if !defined(_WIN32)
#error ERROR: Only Win32 target supported by MIDIFILE.DLL!
#endif

#ifdef __cplusplus
extern "C" {
#endif

// DLL Version #
#define MIDIFILEVERSION 6

#if !defined(_MIDIFILELIB_)
#define MIDIFILEAPI	__declspec( dllimport )
#else
#define MIDIFILEAPI	__declspec( dllexport )
#endif

/* =========================================================================================
 MIDICALL structure -- allocated and initialized by an app, and passed to the MIDIFILE.DLL
 within the MIDIFILE structure. This contains pointers to various functions within the app to
 handle processing of parts of a MIDI file. The MIDIFILE.DLL calls one of those functions when
 handling its respective part of a MIDI file.
 */

typedef struct _MIDICALL
{
	/*	If reading, this reads the specified number of bytes into the specified
		buffer. If writing, this writes the specified number of bytes into the
		specified buffer. If 0, the DLL takes care of reading/writing bytes */
	long (APIENTRY *ReadWriteMidi)();
	
	/*	Opens MIDI file for reading or writing. If 0, then it's assumed that the
		DLL will take care of opening the file, and the MIDIFILE structure's
		Handle field contains a pointer to the filename to open */
	long (APIENTRY *OpenMidi)();
	
	/*	Seeks forward/backward the specified number of bytes from current position.
		If 0, then the DLL takes care of seeking */
	long (APIENTRY *SeekMidi)();
	
	/*	Closes MIDI file. If 0, DLL takes care of closing the file */
	long (APIENTRY *CloseMidi)();

	/*	If reading, called upon load of an MThd header. If writing, initializes the
		Format, NumTracks, and Division fields of MIDIFILE struct (after which
		the DLL then writes out the MThd chunk). */
	long (APIENTRY *StartMThd)();

	/*	If reading, called upon load of an MTrk header. If writing, the app can use
		this to write app-specific chunks before each MTrk, and set up global
		variables for a particular track. Alternately, the app can use this to have
		the DLL write out a pre-formatted MTrk chunk */
	long (APIENTRY *StartMTrk)();

	/*	If reading, called upon load of an unknown header. If writing, the app
		should write out the desired app-specific chunk(s) using the DLL's
		MidiWriteHeader(), 1 or more calls to MidiWriteBytes(), and finally
		MidiCloseChunk(). Such chunks appear after all of the MTrks. */
	long (APIENTRY *UnknownChunk)();

	/*	If reading, loads in and processes text-based Meta-Event via
		MidiReadBytes(). If writing, writes out the remainder of the Meta-event.
		NOTE: Proprietary Meta-Events (type=0x7F) as well as unknown
		Meta-events are also read/written via this function. */
	long (APIENTRY *MetaText)();

	/*	For reading, loads in and processes the remainder of the SYSEX event via
		MidiReadBytes(). For writing, writes out the remainder of the SYSEX
		event via MidiWriteBytes(). */
	long (APIENTRY *SysexEvt)();
	
	/*	If reading, processes loaded MIDI event with status of 0x80 to 0xEF. If
		writing 1 event at a time, reports back to the DLL what event to write */
	long (APIENTRY *StandardEvt)();
	
	/*	If reading, processes loaded Sequence Number Meta-Event. If writing,
		returns the METASEQ formatted for a Sequence Number event */
	long (APIENTRY *MetaSeqNum)();

	/*	Read only. Processes loaded Time Signature Meta-Event */
	long (APIENTRY *MetaTimeSig)();
	
	/*	Read only. Processes loaded Key Signature Meta-Event */
	long (APIENTRY *MetaKeySig)();
	
	/*	Read only. Processes loaded Tempo Meta-Event */
	long (APIENTRY *MetaTempo)();
	
	/*	Read only. Processes loaded SMPTE Meta-Event */
	long (APIENTRY *MetaSMPTE)();
	
	/*	Read only. Processes loaded End Of Track Meta-Event */
	long (APIENTRY *MetaEOT)();
	
	/*	Read only. Processes loaded Port Number Meta-Event */
	long (APIENTRY *MetaPort)();

	/*	Read only. Processes loaded MIDI Channel Meta-Event */
	long (APIENTRY *MetaChan)();
} MIDICALL;

#define CALLREADWRITEMIDI	0
#define CALLOPENMIDI		1
#define CALLSEEKMIDI		2
#define CALLCLOSEMIDI		3
#define CALLSTARTMTHD		4
#define CALLSTARTMTRK		5
#define CALLUNKNOWNCHUNK	6
#define CALLMETATEXT		7
#define CALLSYSEXEVT		8
#define CALLSTANDARDEVT		9
#define CALLMETASEQNUM		10
#define CALLMETATIMESIG		11
#define CALLMETAKEYSIG		12
#define CALLMETATEMPO		13
#define CALLMETASMPTE		14
#define CALLMETAEOT			15
#define CALLMETAPORT		16
#define CALLMETACHAN		17

/* ============================================================================
MIDIFILE structure -- allocated by an app, and passed to the MIDIFILE.DLL in order
for the DLL to help the app read/write MIDI files). This is also passed to each
C MIDICALL. (If a C++ class declares this as the first variable in the class, then
this works out to be the hidden _this pointer passed to your callbacks).
*/

typedef struct _MIDIFILE
{
	/*	Pointer to the MidiCalls structure */
	MIDICALL *		MidiCalls;

	/*	File Handle of the open MIDI file */
	HANDLE			Handle;

	/*	Size of the file. Initially, when the MThd header is read in, this reflects
		the number of data bytes in the remainder of the file (ie, after the 8
		byte MThd header -- starting at the MThd's Format). As bytes are read
		in, this is decremented to reflect how many bytes remain in the file.
		For writes, this reflects how many bytes the DLL has written out (ie,
		size of the MIDI file written by the DLL). */
	long			FileSize;

	/*	ID of the chunk. */
	unsigned long	ID;

	/*	Size of the chunk (MTrk or MThd) currently being read. Initially, when
		the chunk header is read in, this would be the size as specified in the
		header. As bytes are read in, this is decremented. For writes, this is
		used for other purposes by the DLL. */
	long			ChunkSize;

	/*	From MThd */
	unsigned short	Format;

	/*	From MThd */
	unsigned short	NumTracks;

	/*	From MThd */
	unsigned short	Division;

	/*	Flag bits */
	unsigned short	Flags;

	/*	The current event's size. This will be non-zero only if the event is
		a SYSEX or Meta-Event of variable length. As bytes are read in, this is
		decremented. For writes, this is used to specify the length of variable
		length Meta-Events and SYSEX */
	long			EventSize;

	/*	Maintained by DLL */
	unsigned long	PrevTime;

	/*	The current event's time, referenced from 0 (instead of from the previous
		event's time as is done with delta-times in the MIDI file) unless
		MIDIDELTA flag is set */
	unsigned long	Time;

	/*	The track # that the currently loaded or written event belongs to. The DLL increments this */
	unsigned char	TrackNum;

	/*	The event's status. For MIDI events, this is the actual MIDI Status.
		0xFF for Meta-Event. 0xF7 is sysex continuation */
	unsigned char	Status;

	/*	For MIDI events (other than SYSEX), this will contain the 2 subsequent
		MIDI data bytes. If there's only 1 data byte, the second byte will be 0xFF.
		For Meta-Events, the first byte will be the Type. For Meta-Events that
		have a fixed length (ie, Sequence Number, End of Track, Tempo. SMPTE,
		Time Signature, and Key Signature), the rest of the bytes will be the
		remainder of that Meta-Event.
		In other words, the DLL automatically loads MIDI events (except SYSEX)
		and fixed length Meta-Events directly into the MIDIFILE structure. For
		Meta-Events of variable length, and SYSEX messages, the DLL loads the
		Status (0xFF if Meta-Event) and Type (if a Meta-Event), and then loads
		the variable length quantity bytes that form the Length of the event,
		setting EventSize to this value. It's up to the app to then load or skip
		the rest of the event's bytes */
	unsigned char	Data[7];

	/*	Maintained by DLL */
	unsigned char	RunStatus;
} MIDIFILE;


/* ============================== MIDIFILE Flags =============================== */

 /* Set if your callback was called during MidiWriteFile() instead of MidiReadFile() */
#define MIDIWRITE 0x8000

/*	Set if you want the DLL to calculate micros per quarter	from the METATEMPO's TempoBPM field
	when writing a Meta-Tempo event */
#define MIDIBPM   0x4000

/*	Set by the DLL when a 0xF0 event is encountered. The DLL doesn't
	clear this until it encounters a MIDI event with status of 0x80
	to 0xEF or SYSTEM COMMON. Use this when reading via your SysexEvt
	callback to help you distinguish SYSEX continuation events from
	ESCAPED events, both of which have a 0xF7 Status. If an ESCAPED
	event, the first data byte that you MidiReadBytes() in your
	SysexEvt callback should be a MIDI REALTIME or SYSTEM
	COMMON status (bit #7 set). A continuation event can ONLY occur
	when this flag is set, and it's very unlikely that the first data byte
	will have bit #7 set (unless it happens to also be 0xF7, unlikely) */
#define MIDISYSEX 0x2000

/*	When reading/writing a Time Signature Meta-Event, the
	denominator isn't expressed as a power of 2, but rather, as the
	real time sig denominator. In other words for 4/4, instead of a
	Denom of 2, it's 4 */
#define MIDIDENOM 0x1000

/*	Time field is delta instead of referenced from 0 */
#define MIDIDELTA 0x0800

/*	Set this if you want MIDI REALTIME to not cancel running
	status. This will yield more compression if you're saving
	REALTIME events (ie, via an ESCAPE event) intermixed with
	regular MIDI messages. But, some software doesn't follow the
	MIDI spec with regard to REALTIME not affecting running
	status, so this is made optional. */
#define MIDIREALTIME 0x0400

/*	Don't alter this if not using your own ReadWriteMidi callback.
	The DLL uses it for an intelligent File I/O buffering */
#define MIDIDIRTY 0x0200

/*	Set if you want the DLL to calculate micros per quarter	from the METATEMPO's TempoBPM2 field
	when writing a Meta-Tempo event */
#define MIDIBPM2  0x0100



/* ============================================================================
METATEMPO structure -- Passed by DLL to the app's MetaTempo callback. Most of the fields
are the same as the app's MIDIFILE structure with a few, noted exceptions. This is just a
redefinition of the MIDIFILE specifically for Tempo Meta-Events in order to fool the compiler
into seeing certain fields that were defined as UCHAR in MIDIFILE as ULONG in METATEMPO.
Used for both reading and writing.
*/

typedef struct _METATEMPO
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	unsigned long	TempoBPM2;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* Meta Type (ie, 0x51) for reads. 0xFF for writes */
	unsigned char	WriteType;	/* Length (3) for reads. 0x51 for writes. */
	unsigned char	Length;		/* Don't use */
	unsigned long	Tempo;		/* Tempo in micros per quarter note */
	unsigned char	TempoBPM;	/* Tempo in Beats Per Minute */
	unsigned char	RunStatus;
} METATEMPO;



/* ============================================================================
METASEQ structure -- Passed by DLL to the app's MetaSeqNum MIDICALL. This is used by both
reading and writing.
*/

typedef struct _METASEQ
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	unsigned char *	NamePtr;	/* For writing, ptr to the null-terminated Track Name to write as a
								Meta-Event following the Sequence Number Meta-Event, or 0 if none */
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* Meta type (0x00) for reads. 0xFF for writes */
	unsigned char	WriteType;	/* Length (2) for reads. Meta type (0x00) for writes */
	unsigned char	Length;		/* Don't use */
	unsigned short	SeqNum;		/* Sequence number */
	unsigned char	UnUsed2, UnUsed3, UnUsed4;
	unsigned char	RunStatus;
} METASEQ;



/* ============================================================================
METASMPTE structure -- Passed by DLL to the app's MetaSMPTE MIDICALL.
*/

typedef struct _METASMPTE
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	long			UnUsed1;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* Meta Type 0x54 for reads. 0xFF for writes. */
	unsigned char	WriteType;	/* Length (5) for reads. Meta type 0x54 for writes */
	unsigned char	Length;		/* Don't use */
	unsigned char	Hours;		/* SMPTE Hours */
	unsigned char	Minutes;	/* SMPTE Minutes */
	unsigned char	Seconds;	/* SMPTE Secs */
	unsigned char	Frames;		/* SMPTE Frames */
	unsigned char	SubFrames;	/* SMPTE SubFrames */
	unsigned char	RunStatus;
} METASMPTE;



/* ============================================================================
METATIME structure -- Passed by DLL to the app's MetaTimeSig MIDICALL.
*/

typedef struct _METATIME
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	long			UnUsed1;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* Meta Type 0x58 for reads. 0xFF for writes */
	unsigned char	WriteType;	/* Length (4) for reads. Meta Type 0x58 for writes */
	unsigned char	Length;		/* Don't use */
	unsigned char	Nom;		/* Time sig nominator */
	unsigned char	Denom;		/* Time sig denominator */
	unsigned char	Clocks;		/* MIDI clocks in metronome click */
	unsigned char	_32nds;		/* number of 32nd notes in 24 MIDI clocks */
	unsigned char	UnUsed2;
	unsigned char	RunStatus;
} METATIME;



/* ============================================================================
METAKEY structure -- Passed by DLL to the app's MetaKeySig MIDICALL.
*/

typedef struct _METAKEY
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	long			UnUsed1;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* For reads, Meta Type 0x59. 0xFF for writes */
	unsigned char	WriteType;	/* Length (2) for reads. meta type 0x59 for writes */
	unsigned char	Length;		/* Don't use */
	char			Key;		/* -7 for 7 flats... 0 for C... 7 for 7 sharps */
	unsigned char	Minor;		/* 0=Major, 1=Minor */
	unsigned char	UnUsed2, UnUsed3, UnUsed4;
	unsigned char	RunStatus;
} METAKEY;



/* ============================================================================
METAEND structure -- Passed by DLL to the app's MetaEOT MIDICALL.
*/

typedef struct _METAEND
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	long			UnUsed1;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* For reads, Meta Type 0x2F. 0xFF for writes */
	unsigned char	WriteType;	/* Length (0) for reads. Meta type 0x2F for writes */
	unsigned char	Length;		/* Don't use */
	unsigned char	UnUsed2, UnUsed3, UnUsed4, UnUsed5, UnUsed6;
	unsigned char	RunStatus;
} METAEND;


/* ============================================================================
METAPORT structure -- Passed by DLL to the app's MetaPort MIDICALL.
*/

typedef struct _METAPORT
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	long			UnUsed1;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* For reads, Meta Type 0x21. 0xFF for writes */
	unsigned char	WriteType;	/* Length (1) for reads. Meta type 0x21 for writes */
	unsigned char	Length;		/* Don't use */
	unsigned char	PortNumber;	/* The Port Number */
	unsigned char	UnUsed3, UnUsed4, UnUsed5, UnUsed6;
	unsigned char	RunStatus;
} METAPORT;

/* ============================================================================
METACHAN structure -- Passed by DLL to the app's MetaChan MIDICALL.
*/

typedef struct _METACHAN
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	long			UnUsed1;
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* For reads, Meta Type 0x20. 0xFF for writes */
	unsigned char	WriteType;	/* Length (1) for reads. Meta type 0x20 for writes */
	unsigned char	Length;		/* Don't use */
	unsigned char	ChanNumber;	/* The MIDI Channel Number */
	unsigned char	UnUsed3, UnUsed4, UnUsed5, UnUsed6;
	unsigned char	RunStatus;
} METACHAN;

/* ============================================================================
METATXT structure -- Used by app when formatting SYSEX and MetaText events for writing.
*/

typedef struct _METATXT
{
	MIDICALL *		MidiCalls;
	HANDLE			Handle;
	long			FileSize;
	unsigned long	ID;
	long			ChunkSize;
	unsigned short	Format;
	unsigned short	NumTracks;
	unsigned short	Division;
	unsigned short	Flags;
	unsigned long	EventSize;	/* Size of buffer to write out */
	unsigned long	PrevTime;
	unsigned long	Time;
	unsigned char	TrackNum;
	unsigned char	Type;		/* 0xFF for MetaText, 0xF0 or 0xF7 for SYSEX */
	unsigned char	WriteType;	/* MetaType for MetaText events. */
	unsigned char	Length;		/* Don't use */
	unsigned char * Ptr;		/* Pointer to buffer to write out */
	unsigned char	UnUsed2;
	unsigned char	RunStatus;
} METATXT;


/* =========================================================================
 * Errors returned by the DLL
 */

#define MIDIERRSUCCESS	0
#define MIDIERRFILE		1	/* Can't open the MIDI file for reading */
#define MIDIERRINFO		2	/* Can't determine the FileSize for reading */
#define MIDIERRNOMIDI	3	/* Tried to read a file that didn't contain a required MThd */
#define MIDIERRREAD		4	/* An error while reading bytes from the file */
#define MIDIERRWRITE	5	/* An error while writing bytes to the file */
#define MIDIERRBAD		6	/* A mal-formed MIDI file -- it's garbage */
#define MIDIERRSEEK		7	/* An error in setting the file pointer while writing a MIDI file or skipping
								chunks while reading */
#define MIDIERRSTATUS	8	/* Encountered running status where it shouldn't be. (mal-formed MTrk) */
#define MIDIERREVENT	9	/* Encountered an unknown Status while reading in an MTrk.
								Not a Meta-Event, but not a MIDI event either. */
#define MIDIERRFILEW	10	/* Can't open the MIDI file for writing */
#define MIDILOADSTR	10		/* Do not use */
#define MIDISAVESTR	11		/* Do not use */
#define MIDIERRNOMEM	12	/* Memory allocation error */
#define MIDIAPPERR		50	/* App defined errors start with this if positive, or can be
								negative numbers (except for -1 for ReadWriteMidi callback) */


/* ==========================================================================
 * The MIDIFILE.DLL functions
 */

/* reading */
MIDIFILEAPI long MidiReadFile(MIDIFILE * mf);
MIDIFILEAPI long MidiReadFile2(MIDIFILE * mf);
MIDIFILEAPI long MidiReadBytes(MIDIFILE * mf, unsigned char * buf, unsigned long count);
MIDIFILEAPI long MidiSkipChunk(MIDIFILE * mf);
MIDIFILEAPI long MidiSkipEvent(MIDIFILE * mf);
MIDIFILEAPI long MidiReadVLQ(MIDIFILE * mf);
MIDIFILEAPI long MidiReadHeader(MIDIFILE * mf);

/* writing */
MIDIFILEAPI long MidiWriteFile(MIDIFILE * mf);
MIDIFILEAPI long MidiWriteFile2(MIDIFILE * mf);
MIDIFILEAPI long MidiWriteBytes(MIDIFILE * mf, unsigned char * buf, unsigned long count);
MIDIFILEAPI long MidiWriteVLQ(MIDIFILE * mf, unsigned long val);
MIDIFILEAPI long MidiWriteHeader(MIDIFILE * mf);
MIDIFILEAPI long MidiCloseChunk(MIDIFILE * mf);
MIDIFILEAPI long MidiWriteEvt(MIDIFILE * mf);

/* misc */
MIDIFILEAPI long MidiSeek(MIDIFILE * mf, long amt);
MIDIFILEAPI long MidiFileSize(MIDIFILE * mf);
MIDIFILEAPI void MidiFlipLong(unsigned char * ptr);
MIDIFILEAPI void MidiFlipShort(unsigned char * ptr);
MIDIFILEAPI BOOL MidiCompareID(unsigned char * id, unsigned char * ptr);
MIDIFILEAPI void MidiCloseFile(MIDIFILE * mf);
MIDIFILEAPI long MidiVLQToLong(unsigned char * ptr, unsigned long * len);
MIDIFILEAPI unsigned long MidiLongToVLQ(unsigned long val, unsigned char * ptr);
MIDIFILEAPI unsigned long MidiGetErr(MIDIFILE * Mf, unsigned char * Buffer, unsigned long BufferSize, long Error);
MIDIFILEAPI unsigned char * MidiGetVers();
MIDIFILEAPI void MidiSetCallback(MIDICALL * MidiCall, long (APIENTRY * Function)(), unsigned long Offset);
MIDIFILEAPI void MidiSetRWCall(MIDICALL * MidiCall, long (APIENTRY * Function)(unsigned char *, unsigned long));
MIDIFILEAPI void MidiSetSeekCall(MIDICALL * MidiCall, long (APIENTRY * Function)(long));

/* Version/Language */
MIDIFILEAPI unsigned char MidiFileVersion[2];

// C++ App must set this to its "this" pointer if using Member functions
MIDIFILEAPI MIDIFILE * MidiApp;

/* ============================= C++ Stuff ================================= */
#ifdef __cplusplus
}

// CMidiFile Class
class CMidiFile
{
public:
	// MidiFile struct
	MIDIFILE	MidiFile;

	// MidiCall struct
	long (APIENTRY CMidiFile:: * ReadWriteMidi)(unsigned char *, unsigned long);
	long (APIENTRY CMidiFile:: * OpenMidi)();
	long (APIENTRY CMidiFile:: * SeekMidi)(long);
	long (APIENTRY CMidiFile:: * CloseMidi)();
	long (APIENTRY CMidiFile:: * StartMThd)();
	long (APIENTRY CMidiFile:: * StartMTrk)();
	long (APIENTRY CMidiFile:: * UnknownChunk)();
	long (APIENTRY CMidiFile:: * MetaText)();
	long (APIENTRY CMidiFile:: * SysexEvt)();
	long (APIENTRY CMidiFile:: * StandardEvt)();
	long (APIENTRY CMidiFile:: * MetaSeqNum)();
	long (APIENTRY CMidiFile:: * MetaTimeSig)();
	long (APIENTRY CMidiFile:: * MetaKeySig)();
	long (APIENTRY CMidiFile:: * MetaTempo)();
	long (APIENTRY CMidiFile:: * MetaSMPTE)();
	long (APIENTRY CMidiFile:: * MetaEOT)();
	long (APIENTRY CMidiFile:: * MetaPort)();
	long (APIENTRY CMidiFile:: * MetaChan)();

	// Constructor
	CMidiFile()
	{
		// Put address of CALLBACK into MIDIFILE
		MidiFile.MidiCalls = (MIDICALL *)&ReadWriteMidi;
	}

	// Destructor
	~CMidiFile() {}
};
#endif /* _cplusplus */

#endif /* _INC_MIDIFILE */
