/* MidSkele -- A minimal Windows C MIDI program
 *
 * This file contains the Save/Load settings to/from the Windows registry
 */

/* Windows includes */
#include <windows.h>
#include <windowsx.h>
#include <winreg.h>

/* app includes */
#include "resource.h"
#include "globals.h"
#include "midi_io.h"





/* Here are my registry key names */
const char		KeyName[] = "Software\\MidiSkeleton";
const char		MidiInKey[] = "MidiInID";
const char		MidiOutKey[] = "MidiOutID";
const char		RelinquishKey[] = "Relinquish";

/* Errors */
const char		BadRegSave[] = "Bad save of settings!";
const char		RegErrorStr[] = "Registry Error";







/**************************** loadSettings() ***************************
 * Loads settings from the Software\MidiSkeleton key under CURRENT_USER
 * section of the Windows 95 registry, and stuffs them into variables.
 ***********************************************************************/

void loadSettings(void)
{
	HKEY		hKey;

	/* Open Software\MidiSkeleton key under CURRENT_USER */
	if (!RegOpenKeyEx(HKEY_CURRENT_USER, &KeyName[0], 0, KEY_ALL_ACCESS, &hKey))
	{
		unsigned long	type, val, len;

		/* Read in a long value */
		val = 4;
		type = REG_DWORD;

		/* Read MidiInID key's value */
		if (!RegQueryValueEx(hKey, &MidiInKey[0], 0, &type, (unsigned char *)&len, &val))
		{
			/* Store ID # */
			MidiInID = len;

			/* Read MidiOutID key's value */
			if (!RegQueryValueEx(hKey, &MidiOutKey[0], 0, &type, (unsigned char *)&len, &val))
			{
				MidiOutID = len;

				/* Read Relinquish key's value */
				if (!RegQueryValueEx(hKey, &MidiOutKey[0], 0, &type, (unsigned char *)&len, &val))
					NoRelinquishFlag = (unsigned char)len;
			}
		}

		/* Close registry key */
		RegCloseKey(hKey);
	}
}





/**************************** saveSettings() ***************************
 * Saves settings into the Software\MidiSkeleton key under CURRENT_USER
 * section of the Windows 95 registry.
 ***********************************************************************/

void saveSettings(void)
{
	HKEY			hKey;
	unsigned long	len;

	/* Delete any previous key with this name */
	RegDeleteKey(HKEY_CURRENT_USER, &KeyName[0]);

	/* Open/Create the Software\MidiSkeleton key under CURRENT_USER */
	if (!RegCreateKeyEx(HKEY_CURRENT_USER, &KeyName[0], 0, (char *)&KeyName[9], REG_OPTION_NON_VOLATILE,
		KEY_ALL_ACCESS, 0, &hKey, &len))
	{
		/* Create the MIDI In Device ID registry key and set its value */
		if (!(len = RegSetValueEx(hKey, &MidiInKey[0], 0, REG_DWORD, (unsigned char *)&MidiInID, 4)))
		{
			/* Create the MIDI Out Device ID registry key and set its value */
			if (!(len = RegSetValueEx(hKey, &MidiOutKey[0], 0, REG_DWORD, (unsigned char *)&MidiOutID, 4)))
			{
				/* Create the Relinquish registry key and set its value */
				len = NoRelinquishFlag;
				if (!(len = RegSetValueEx(hKey, &RelinquishKey[0], 0, REG_DWORD, (unsigned char *)&len, 4)))
				{
			
					/* Close MidiSkeleton key */
					RegCloseKey(hKey);

					/* Success */
					return;
				}
			}
		}

		/* Close MidiSkeleton key */
		RegCloseKey(hKey);
	}

	/* Error */
	MessageBox(MainWindow, &BadRegSave[0], &RegErrorStr[0], MB_OK|MB_ICONEXCLAMATION);
}
