// MfcMidiFile.h : header file

// Uncomment this define if you want standard C buffered file I/O
// #define BUFFERED_IO

// Uncomment this define if you want an example of writing a MIDI file into a RAM buffer
// #define RAMFILE_IO

// Include the MIDIFILE.DLL header file. It defines the CMidiFile base class
#include "..\midifile.h"

/*	Let's define some "event" structures for the internal use of this program. Each will hold one
	"event". The first ULONG will be the event's time (referenced from 0). Then, there will be 4
	more bytes. The first of those 4 will be the event's Status, and the remaining 3 will be
	interpreted differently depending upon the Status. By having all of the events the same size
	(ie, 8 bytes), with the Time and Status first, this makes it easy to sort, insert, and delete
	events from a block of memory. The only drawback is that 8 bytes is a little cramped to store
	lots of info, so you could go with a larger structure, but 8 bytes is perfect for the majority of
	events.
	For fixed length MIDI events (ie, Status is 0x80 to 0xEF, 0xF1, 0xF2, 0xF3, 0xF6, 0xF8,
	0xFA, 0xFB, 0xFC, and 0xFE), the Status is simply the MIDI status, and the subsequent Data
	bytes are any subsequent MIDI data bytes. For example, for a Note-On, Status would be 0x90
	to 0x9F, Data1 would be the note number, Data2 would be the velocity, and Data3 wouldn't be
	used.
	For Meta-Events, the Status will be the meta Type (ie, instead of 0xFF). Note that all
	Meta-Event types are less than 0x80, so it's easy to distinguish them from MIDI status
	bytes. For fixed length Meta-Events (ie, meta types of 0x00, 0x2F, 0x51, 0x54, 0x58, and
	0x59), the remaining 3 bytes will be interpreted as follows:
			TYPE			DATA BYTES 1, 2, and 3
		Sequence Number	=	Not Used, SeqNumLSB, SeqNumMSB
		End Of Track	=	Not Used...
		Tempo			=	BPM, Not Used, Not Used
		SMPTE			=	There's too much data to store in only 3 bytes. We could define
							the bytes to be an index into a list of larger structures, but I'll
							just ignore SMPTE for simplicity.
		Time Signature	=	Numerator, Denominator, MIDI clocks per metronome click.
		Key Signature	=	Key, Minor, Not Used
		Port Number		=	Port, Not Used, Not Used
		MIDI Channel Number	=	Channel, Not Used, Not Used
	For variable length Meta-Events (ie, type of 0x01 to 0x0F, or 0x7F), since these have data
	that can be any length, we'll set Data1 to be the length of the data (ie, we'll set a limit of
	255 bytes), and Data2 and Data3 will together form a USHORT index into a larger array
	(ie, we'll set a limit of 65,535 such Meta-Events).
	For SYSEX events (0xF0 or 0xF7), these also can be any length, so we'll set Data1 to
	be an index into a larger array (ie, set a limit of 256 such events), and Data2 and Data3
	will form a USHORT length of the SYSEX (not counting the Status).
*/

typedef struct _EVENT  // general form of an "event"
{
	unsigned long Time;
	unsigned char Status;
	unsigned char Data1, Data2, Data3;
} EVENT;

typedef struct _XEVENT
{
	unsigned long  Time;
	unsigned char  Status;
	unsigned char  Index;
	unsigned short Length;
} XEVENT;

typedef struct _SEQEVENT
{
	unsigned long  Time;
	unsigned char  Status;
	unsigned char  UnUsed1;
	unsigned short SeqNum;
} SEQEVENT;

typedef struct _TXTEVENT
{
	unsigned long  Time;
	unsigned char  Status;
	unsigned char  Length;
	unsigned short Index;
} TXTEVENT;

typedef struct _TEMPOEVENT
{
    unsigned long  Time;
    unsigned char  Status;
    unsigned char  BPM, Unused1, UnUsed2;
} TEMPOEVENT;

typedef struct _TIMEEVENT
{
    unsigned long Time;
    unsigned char Status;
    unsigned char Nom, Denom, Clocks;
} TIMEEVENT;

typedef struct _KEYEVENT
{
    unsigned long Time;
    unsigned char Status;
    unsigned char Key, Minor, UnUsed1;
} KEYEVENT;

typedef struct _PORTEVENT
{
    unsigned long Time;
    unsigned char Status;
    unsigned char Port, Unused1, UnUsed2;
} PORTEVENT;

typedef struct _CHANEVENT
{
    unsigned long Time;
    unsigned char Status;
    unsigned char Chan, Unused1, UnUsed2;
} CHANEVENT;




////////////////////////////////////////////////////////////////////////////
// Definition of MfcMidiFile class
class MfcMidiFile: public CMidiFile
{
public:
	MfcMidiFile();
	void 			SaveMidi(const char * Filename, unsigned char Format);

private:
	void			OnStoreSeq(SEQEVENT * trk, METASEQ * mf);
	void			OnStoreTempo(TEMPOEVENT * trk, METATEMPO * mf);
	void			OnStoreTimeSig(TIMEEVENT * trk, METATIME * mf);
	void			OnStoreKey(KEYEVENT * trk, METAKEY * mf);
	void			OnStoreMeta(TXTEVENT * trk, METATXT * mf);
	void			OnStoreSysex(XEVENT * trk, METATXT * mf);
	void			OnStorePort(PORTEVENT * trk, METAPORT * mf);
	void			OnStoreChan(CHANEVENT * trk, METACHAN * mf);

	// The Callbacks
#ifdef BUFFERED_IO
	long APIENTRY	OnReadWriteMidi(unsigned char * Buffer, unsigned long Count);
	long APIENTRY	OnOpenMidi();
	long APIENTRY	OnSeekMidi(long Offset);
	long APIENTRY	OnCloseMidi();
#endif
#ifdef RAMFILE_IO
	long APIENTRY	OnReadWriteMidi(unsigned char * Buffer, unsigned long Count);
	long APIENTRY	OnOpenMidi();
	long APIENTRY	OnSeekMidi(long Offset);
	long APIENTRY	OnCloseMidi();
#endif
	long APIENTRY	OnStartMTrk();
	long APIENTRY	OnUnknownChunk();
	long APIENTRY	OnSysexEvt();
	long APIENTRY	OnStandardEvt();
	long APIENTRY	OnMetaSeqNum();
};


// An instance of MfcMidiFile (found in MfcMidiFile.cpp)
extern MfcMidiFile	MyCMidiFile;
