// MfcMidiFile.cpp : implementation file
// This contains most all of the relevant stuff concerning writing a MIDI file using MIDIFILE.DLL

#include <sys/types.h>
#include <sys/stat.h>
#include "stdafx.h"
#include "MfcMidiFile.h"
#include "..\midifile.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif





// An Instance of MfcMidiFile
MfcMidiFile		MyCMidiFile;





// OK, here's an artificially created block of EVENTS. (Check MfcMidiFile.h for the definition
// of an EVENT structure. I mix it up with a variety of events, to show you how you might store
// such. This is for writing a Format 0. In your own app, you may choose to store data in a different way.

EVENT evts[21] = {
     {0, 0x04, 5, 0x01, 0x00},	// Instrument Name Meta-Event. Status = 0x04. Note that Data2
								// and Data3 form a USHORT index. Because Intel CPU uses
								// reverse byte order, the seq number here is really 0x0001
     {0, 0x01, 11, 0x00, 0x00}, // A Text Meta-Event. Status = 0x01
     {0, 0x58, 5, 4, 24},		// A Time Signature Meta-Event. Status = 0x58. 5/4
     {0, 0x59, 1, 0, 0},		// A Key Signature Meta-Event. Status = 0x59. G Major
     {0, 0x51, 100, 0, 0},		// A Tempo Meta-Event. Status = 0x51
     {0, 0x21, 0, 0, 0},		// A Port Number Meta-Event. Status = 0x21. This is converted to a "Device Name" meta-event by MIDIFILE
     {96, 0x90, 64, 127, 0},	// A Note-on (ie, fixed length MIDI message)
     {96, 0xC0, 1, 0xFF, 0},	// A Program Change (ie, also a fixed length MIDI message)
     {96, 0xF0, 0, 0x02, 0x00},	// SYSEX (ie, 0xF0 type)
     {96, 0x90, 71, 127, 0},	// A Note-on (ie, fixed length MIDI message)
     {192, 0xFC, 0, 0, 0},		// MIDI REALTIME "Stop"
     {192, 0x90, 68, 127, 0},	// A Note-on (ie, fixed length MIDI message)
     {288, 0x90, 64, 0, 0},		// A Note-off (ie, Note-on with 0 velocity)
     {288, 0x51, 120, 0, 0},	// A Tempo Meta-Event. Status = 0x51
     {288, 0x90, 68, 0, 0},		// A Note-off
     {384, 0xF0, 1, 0x01, 0x00}, // SYSEX (ie, 0xF0 type, but the start of a series of packets,
								//	so it doesn't end with 0xF7)
     {384, 0xF7, 2, 0x01, 0x00}, // SYSEX (ie, 0xF7 type, the next packet)
     {384, 0xF7, 3, 0x02, 0x00}, // SYSEX (ie, 0xF7 type, the last packet, so it ends with 0xF7)
     {480, 0x7F, 4, 0x02, 0x00}, // Proprietary Meta-Event
     {480, 0x90, 71, 0, 0},		// A Note-off
     {768, 0x2F, 0, 0, 0},		// An End Of Track Meta-Event. Status = 0x2F
};





// Here's 2 artificially created blocks of EVENTS. These 2 are for writing a Format 1. We simply
// separate the events in the above track into 2 tracks. With a format 1, the first track is
// considered the "tempo map", so we put appropriate events in that (ie, all tempo. smpte, and time
// signature events should go in only this one track).

EVENT trk1[6] = {
     {0, 0x58, 5, 4, 24},
     {0, 0x59, 1, 0, 0},
     {0, 0x51, 100, 0, 0},
     {288, 0x51, 120, 0, 0},
     {480, 0x7F, 4, 0x02, 0x00},
     {768, 0x2F, 0, 0, 0},
};

EVENT trk2[16] = {
     {0, 0x04, 5, 0x01, 0x00},
     {0, 0x01, 11, 0x00, 0x00},
     {0, 0x21, 0, 0, 0},
     {96, 0x90, 64, 127, 0},
     {96, 0xC0, 1, 0xFF, 0},
     {96, 0xF0, 0, 0x02, 0x00},
     {96, 0x90, 71, 127, 0},
     {192, 0xFC, 0, 0, 0},
     {192, 0x90, 68, 127, 0},
     {288, 0x90, 64, 0, 0},
     {288, 0x90, 68, 0, 0},
     {384, 0xF0, 1, 0x01, 0x00},
     {384, 0xF7, 2, 0x01, 0x00},
     {384, 0xF7, 3, 0x02, 0x00},
     {480, 0x90, 71, 0, 0},
     {768, 0x2F, 0, 0, 0},
};





// Pointers to our tracks. Assume Format 0. In your own app, you may choose to do something different
EVENT * trk_ptrs[2] = { &evts[0], &trk2[0] };





// An array of text for Meta-Text events. In your own app, you may choose to do something different
unsigned char text[3][20] = {
     { "Here's text" },  // For Text
     { "Piano" },       // For Instrument Name
     { 0, 1, 2, 3 },	// For Proprietary
};





// An array of track names. In your own app, you may choose to do something different
unsigned char names[2][20] = {
     { "Dummy" },
     { "Blort" },
};





// An array of SYSEX data. In your own app, you may choose to do something different
unsigned char sysex[4][2] = {
     { 0x01, 0xF7 },   // an 0xF0 type always ends with 0xF7 unless it has an 0xF7 type to follow
     { 0x02 },
     { 0x03 },		// an 0xF7 type doesn't always end with 0xF7 unless it's the last packet
     { 0x04, 0xF7 },
};





// To point to the next event to write out
EVENT * TrkPtr;





// To look up the address of sysex data
unsigned char ex_index;





// **************************** SaveMidi() ***************************
// Called when user clicks on the "Save" button.

void MfcMidiFile::SaveMidi(const char * Filename, unsigned char Format)
{
	long		result;
	char		buf[80];

	// Store a pointer to the filename in MIDIFILE's Handle field
	MidiFile.Handle = (HANDLE)Filename;

	// Make it easier for me to specify Tempo and Time Signature events
	MidiFile.Flags = MIDIBPM|MIDIDENOM;

	// NOTE: We can initialize the MThd before the call to MidiWriteFile(), or we can do it in
	// our StartMThd callback. We'll do it now since we don't need to do anything else right
	// before the MThd is written out (ie, and so don't need a StartMThd callback)

	// Assume Format 0
	MidiFile.NumTracks = 1;
	trk_ptrs[0] = &evts[0];
	if ((MidiFile.Format = Format))
	{
		// Format 1 or 2. Do 2 MTrks
		MidiFile.NumTracks = 2;
		trk_ptrs[0] = &trk1[0];
	}

	// Arbitrarily chose 96 PPQN for my time-stamps
	MidiFile.Division = 96;

	// Set the DLL Variable MidiApp
	MidiApp = (MIDIFILE *)this;

	// Tell MIDIFILE.DLL to write out the file, calling my callback functions
	result = MidiWriteFile2(&MidiFile);

	// Print out error message. (NOTE: For 0, DLL returns a "Successful MIDI file save" message).
	if (result)
	{
		// See if it's one of the DLL's errors. If not the return will be 0
		if (!MidiGetErr(&MidiFile, (unsigned char *)&buf[0], 80, result))
		{
			// Must be an error that one of my callbacks returned
			strcpy(&buf[0], "My callback returned error #");
			_itoa(result, &buf[28], 10);
		}

		// Display the error message
		AfxMessageBox((LPCTSTR)&buf[0], MB_OK|MB_ICONEXCLAMATION, (unsigned long)result);
	}
}










// ****************************** OnStartMTrk() *****************************
// * This is called by MIDIFILE.DLL before it writes an MTrk chunk's header (ie,
// * before it starts writing out an MTrk). The MIDIFILE's TrackNum field is
// * maintained by the DLL, and reflects the MTrk number (ie, the first MTrk is
// * 0, the second MTrk is 1, etc). We could use this to help us "look up" data
// * and structures associated with that track number. (It's best not to alter
// * this MIDIFILE field). At this point, we should NOT do any MidiReadBytes()
// * because the header hasn't been written yet. What we perhaps can do is
// * write out some chunk of our own creation before the MTrk gets written. We
// * could do this with MidiWriteHeader(), MidiWriteBytes(), and then
// * MidiCloseChunk(). But, that's probably not a good idea. There are many
// * programs that have a very inflexible way of reading MIDI files. Such
// * software might barf on a chunk that isn't an MTrk, even though it's
// * perfectly valid to insert such chunks. The DLL correctly handles such
// * situations. In fact, the MIDICALL allows you to specify a function for
// * chunks that aren't MTrks.
// *
// * This should return 0 to let the DLL write the MTrk header, after which the
// * DLL starts calling our StandardEvt() callback (below) for each event that
// * we wish to write to that MTrk chunk (up to an End Of Track Meta-Event).
// *
// * Alternately, if we already had all of the MTrk data in a buffer (minus the
// * MTrk header), formatted and ready to be written out as an MTrk chunk, we could
// * place the size and a pointer to that buffer in the MIDIFILE's EventSize and
// * Time fields respectively, and return 0. In this case, the DLL will write out
// * the MTrk chunk completely and then call our StandardEvt() callback once.
// *
// * If this function returns non-zero, that will cause the write to abort and
// * we'll return to main() after the call to MidiWriteFile(). The non-zero value
// * that we pass back here will be returned from MidiWriteFile().
// **************************************************************************

long APIENTRY MfcMidiFile::OnStartMTrk()
{
	// Initialize a global ptr to the start of the EVENTs that we're going to write in this MTrk. Our
	// OnStandardEvt callback will use this to locate the next event to be written
	TrkPtr = trk_ptrs[MidiFile.TrackNum];

	// Here, we could write out some non-standard chunk of our own creation. It will end up
	// in the MIDI file ahead of the MTrk that we're about to write. Nah!

	// Return 0, but we didn't place our own buffer ptr into MIDIFILE's Time field and the size of
	// the data to be written in MIDIFILE's EventSize field, so the DLL will instead call
	// OnStandardEvt() for each event to be written
	return(0);
}





// ******************************* OnStoreSeq() *****************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a SEQUENCE
// * NUMBER (and SEQUENCE NAME) Meta-Events
// **************************************************************************

void MfcMidiFile::OnStoreSeq(SEQEVENT * trk, METASEQ * mf)
{
	mf->NamePtr = &names[mf->TrackNum][0];
	mf->SeqNum = trk->SeqNum;
}





// ******************************* OnStoreTempo() ****************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a TEMPO
// * Meta-Event
// ***************************************************************************

void MfcMidiFile::OnStoreTempo(TEMPOEVENT * trk, METATEMPO * mf)
{
	// NOTE: We set MIDIBPM Flag so the DLL will calculate micros from this BPM.
	// Without MIDIBPM, we'd have to set mf->Tempo to the micros, and ignore mf->TempoBPM
	mf->TempoBPM = trk->BPM;
}





// ****************************** OnStoreTimeSig() ***************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a TIME
// * SIGNATURE Meta-Event
// ***************************************************************************

void MfcMidiFile::OnStoreTimeSig(TIMEEVENT * trk, METATIME * mf)
{
	mf->Nom = trk->Nom;
	mf->Denom = trk->Denom;
	mf->Clocks = trk->Clocks;
	mf->_32nds = 8;		// hard-wire this to 8 for my app. You might do it differently
}





// ******************************* OnStoreKey() ******************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a KEY
// * SIGNATURE Meta-Event
// ***************************************************************************

void MfcMidiFile::OnStoreKey(KEYEVENT * trk, METAKEY * mf)
{
	mf->Key = trk->Key;
	mf->Minor = trk->Minor;
}




// ****************************** OnStorePort() ******************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a PORT
// * NUMBER Meta-Event
// ***************************************************************************

void MfcMidiFile::OnStorePort(PORTEVENT * trk, METAPORT * mf)
{
	mf->PortNumber = trk->Port;
}





// ****************************** OnStoreChan() ******************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a MIDI
// * CHANNEL NUMBER Meta-Event
// ***************************************************************************

void MfcMidiFile::OnStoreChan(CHANEVENT * trk, METACHAN * mf)
{
	mf->ChanNumber = trk->Chan;
}





// ******************************* OnStoreMeta() ******************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out one of the
// * variable length Meta-Events (ie, types 0x01 to 0x0F, or 0x7F).
// ****************************************************************************

void MfcMidiFile::OnStoreMeta(TXTEVENT * trk, METATXT * mf)
{
	// If a variable length Meta-Event (ie, type is 0x01 to 0x0F, or 0x7F), then we set
	// the EventSize to the number of bytes that we expect to output. We also set the
	// Ptr (ie, ULONG at Data[2]) either to a pointer to a buffer containing the data bytes
	// to write, or 0. If we set a pointer, then when we return, the DLL writes the data,
	// and then calls StandardEvt for the next event. If we set a 0 instead, when we return,
	// the DLL will call our MetaText callback next, which is expected to MidiWriteBytes() that
	// data. Here, we'll supply the pointer and let the DLL do all of the work of writing out
	// the data

	// Look up where I've stored the data for this meta-event (ie, in my text[] array), and
    // store this pointer in the MIDIFILE
	mf->Ptr = &text[trk->Index][0];

	// Set the event length. NOTE: If we set this to 0, then that tells the DLL that
	// we're passing a null-terminated string, and the DLL uses strlen() to get the length.
	// We could have done that here to really simplify the structure of our TXTEVENT
	// since all of our strings happen to be null-terminated. But, if you ever need to write
	// out data strings with imbedded NULLs, you'll need to do something like this...
	mf->EventSize = (unsigned long)trk->Length;
}





// ****************************** OnStoreSysex() *****************************
// * Called by OnStandardEvt() to format the MIDIFILE for writing out a SYSEX
// * Meta-Events
// **************************************************************************

void MfcMidiFile::OnStoreSysex(XEVENT * trk, METATXT * mf)
{
	// For SYSEX, just store the status, and the length of the message. We also set the
	// the ULONG at Ptr (ie, Data[2]), just like with variable length Meta-Events. (See comment
	// above). Upon return, the DLL will write the supplied buffer, or if none supplied,
	// call our SysexEvt callback which will MidiWriteBytes() the rest of the message.
	// Here, let's do the opposite of what we did in store_meta(), just for the sake
	// of illustration. The easier way is to let the DLL write out the SYSEX data, if that
	// data happens to be in one buffer.

	// Set the length
	mf->EventSize = (unsigned long)trk->Length;

	// Store index in a global for our SysexEvt callback. Alternately, we could put it
	// into the UnUsed2 field of the METATXT (since it's a UCHAR), and retrieve the
	// value in OnSysexEvt()
	ex_index = trk->Index;

	// Set Ptr (ie, ULONG at Data[2]) to 0. This ensures that the DLL calls OnSysexEvt instead
	// of writing out a buffer for us
	mf->Ptr = 0;
}





// ******************************* OnStandardEvt() *******************************
// * This is called by MIDIFILE.DLL for each event to be written to an MTrk chunk
// * (or it's only called once if we supplied a preformatted buffer in our StartMTrk
// * callback). This does the real work of feeding each event to the DLL to write
// * out within an MTrk.
// *
// * The MIDIFILE's TrackNum field is maintained by the DLL, and reflects the MTrk
// * number. We could use this to help us "look up" data associated with that track
// * number.
// *
// * We need to place the event's time (referenced from 0 as opposed to the previous
// * event unless we set the MIDIDELTA Flag) in the MIDIFILE's Time field. When we
// * return from here, the DLL will write it out as a variable length quantity.
// * Next, we need to place the Status in the MIDIFILE's Status field. For fixed
// * length MIDI events (ie, Status < 0xEF), then this is simply the MIDI Status
// * byte. No running status. We must provide the proper status if it's a fixed
// * length MIDI message. Then, we place the 1 or 2 subsequent MIDI data bytes for
// * this event in the MIDIFILE's Data[0] and Data[1]. Upon return, the DLL will
// * completely write out the event, and then call this function again for the next
// * event.
// *
// * For fixed length Meta-Events (ie, meta types of 0x00, 0x21, 0x2F, 0x51, 0x54,
// * 0x58, and 0x59), the Status must be 0xFF, and Data[0] must be the meta type.
// * The characters starting at Data[1] (ie, length byte) must be the rest of the
// * message. Upon return, the DLL will completely write out the event, and then
// * call this function again for the next event. Note that the DLL automatically
// * set the length for this fixed length Meta-Events, so you need not bother setting
// * Data[1].
// *
// * For a Tempo Meta-Event, you have an option. Although Status=0xFF and
// * Data[0]=0x51 always, instead of placing the micros per quarter as a ULONG
// * starting at Data[2] (ie, Data[2], Data[3], Data[4], and Data[5]), you can
// * alternately place the tempo BPM in Data[6] and set the MIDIFILE's Flags MIDIBPM
// * bit. Note that if you recast the MIDIFILE as a METATEMPO, it makes it easier to
// * store the micros as a ULONG in the Tempo field, and BPM in the Tempo field. The
// * DLL will format the micros bytes for you.
// *
// * When you return with an End Of Track event (ie, Status=0xFF and Data[0]=0x2f),
// * then the DLL will write out this event and close the MTrk chunk. It then moves
// * on to the next MTrk if there is another, calling OnStartMTrk again with the next
// * TrackNum.
// *
// * For SYSEX events (ie, Status = 0xF0 and 0xF7), you set the status byte and then
// * set EventSize to how many bytes are in the message (not counting the status). You
// * then can set the ULONG at Data[2] (ie, Data[2], Data[3], Data[4], and Data[5]) to
// * point to a buffer containing the remaining bytes (after 0xF0 or 0xF7) to output.
// * Alternately, you can set this ULONG to 0. The DLL will write out the status and
// * length as a variable length quantity. Then, if you supplied the buffer pointer,
// * it will write out the data. If you set the pointer to 0, then the DLL will call
// * your SysexEvt callback which is expected to MidiWriteBytes() the actual data
// * for this SYSEX message. Note that the SysexEvt could make numerous calls to
// * MidiWriteBytes, before it returns. The number of bytes written must be the same
// * as the previously specified EventSize.
// *
// * For variable length Meta-Events (ie, types of 0x01 through 0x0F, and 0x7F), you
// * only set the Status=0xFF, Data[0] = the meta type, and then set EventSize to how
// * many bytes are in the message (not counting the status and type). You can then
// * set the ULONG at Data[2]. (See the notes on SYSEX above). The DLL will write out
// * the status, type, and length as a variable length quantity, and then either the
// * supplied buffer, or if no supplied buffer, the DLL will call your MetaText
// * callback which is expected to MidiWriteBytes() the actual data for this Meta-Event.
// *
// * For MIDI REALTIME and SYSTEM COMMON messages (ie, 0xF1, 0xF2, 0xF3, 0xF6, 0xF8,
// * 0xFA, 0xFB, 0xFC, and 0xFE), simply set the MIDIFILE's Status to the appropriate
// * MIDI Status, and then store any subsequent data bytes (ie, 1 or 2) in Data[0] and
// * Data[1]. Upon return, the DLL will completely write out the event, and then call
// * this function again for the next event.
// *
// * If this function returns non-zero, that will cause the write to abort and we'll
// * return to main() after the call to MidiWriteFile(). The non-zero value that we
// * pass back here will be returned from MidiWriteFile().
// *********************************************************************************

long APIENTRY MfcMidiFile::OnStandardEvt()
{
	register unsigned char chr;

again:
	// Store the Time of this event in MIDIFILE's Time field
	MidiFile.Time = TrkPtr->Time;

	// Get the Status for this event
	chr = MidiFile.Status = TrkPtr->Status;

	// Is this a Meta-Event? (ie, we use meta's Type for the status, and these values are always
	// less than 0x80 -- therefore bit #7 clear)
	if ( !(chr & 0x80) )
	{
		// Set Status to 0xFF
		MidiFile.Status = 0xFF;

		// Set Data[0] = Type
		MidiFile.Data[0] = chr;

		switch (chr)
		{
			// NOTE: MIDIFILE.DLL sets Data[1] to the proper length for the fixed length meta
			// types (ie, 0x00, 0x2F, 0x51, 0x54, 0x58, and 0x59) so we don't need to do that

			// ------- Sequence # -------
			// NOTE: If we use a MetaSeqNum callback, we wouldn't write out a Meta-Event
			// here, and so this case wouldn't be needed
			case 0x00:
				// Note the recasting of the EVENT and the MIDIFILE structures to the versions
				// appropriate for a SEQUENCE NUMBER Meta-Event. I use this technique several
				// times below
				OnStoreSeq( (SEQEVENT *)TrkPtr, (METASEQ *)&MidiFile );
				break;

			// ---- MIDI Channel Number ---
			case 0x20:
				OnStoreChan( (CHANEVENT *)TrkPtr, (METACHAN *)&MidiFile );
				break;

			// ------- Port Number ------
			case 0x21:
				OnStorePort( (PORTEVENT *)TrkPtr, (METAPORT *)&MidiFile );
				break;

			// ------- End of Track ------
	      	case 0x2F:
				break;

			// ------- Set Tempo --------
			case 0x51:
				OnStoreTempo( (TEMPOEVENT *)TrkPtr, (METATEMPO *)&MidiFile );
				break;

			// --------- SMPTE ---------
			case 0x54:
				// Right now, let's ignore SMPTE events. This was too much of a hassle
				// since there are too many data bytes to fit into the 8 bytes that I use to
				// express events internally in this program

				// Advance the ptr to the next event (ie, for the next call to this function)
				TrkPtr++;

				// Format next event
				goto again;

			// ------ Time Signature -----
			case 0x58:
				OnStoreTimeSig( (TIMEEVENT *)TrkPtr, (METATIME *)&MidiFile );
				break;

			// ------ Key Signature ------
			case 0x59:
				OnStoreKey( (KEYEVENT *)TrkPtr, (METAKEY *)&MidiFile );
				break;

			// Must be a variable length Meta-Event (ie, type is 0x01 to 0x0F, or 0x7F)
			default:
				OnStoreMeta( (TXTEVENT *)TrkPtr, (METATXT *)&MidiFile );
		}
	}

	// Must be a real MIDI event (as opposed to a MetaEvent)
	else switch ( chr )
	{
		// SYSEX (0xF0) or SYSEX CONTINUATION (0xF7)
		case 0xF0:
		case 0xF7:
			OnStoreSysex( (XEVENT *)TrkPtr, (METATXT *)&MidiFile );
			break;

		// For other MIDI messages, they're all fixed length, and will fit into the MIDIFILE
		// structure, so we copy 2 more data bytes to the MIDIFILE (whether those data bytes
		// are valid or not -- the DLL takes care of writing the message out properly)
		default:
			MidiFile.Data[0] = TrkPtr->Data1;
			MidiFile.Data[1] = TrkPtr->Data2;
	}

	// Advance the ptr to the next event (ie, for the next call to this function)
	TrkPtr++;

	return(0);
}





// ******************************** OnSysexEvt() ******************************
// * This is called by MIDIFILE.DLL if it needs me to write out the data bytes
// * for a SYSEX message being written to an MTrk chunk. If the DLL called me,
// * then OnStandardEvt must have initiated writing a SYSEX event to an MTrk, but
// * didn't supply a pointer to a buffer filled with data. Now, I need to write
// * those data bytes here. NOTE: EventSize still contains the length of
// * data to write.
// ****************************************************************************

long APIENTRY MfcMidiFile::OnSysexEvt()
{
	register unsigned char *	ptr;
//	register unsigned long		i;
//	register long				result;

	// Look up where I put the data for the SYSEX event that we're writing right now
	ptr = &sysex[ex_index][0];

	// Here's an example of writing the data one byte at a time. This is slow, but it shows that
	// you can make multiple calls to MidiWriteBytes(). NOTE: commented out; just for
	// illustration. Also note that MidiWriteBytes() decrements EventSize, so we could have
	// done: while( MidiFile.EventSize )
//	for (i = MidiFile.EventSize; i; i--)
//	{
//		if ( (result = MidiWriteBytes(&MidiFile, ptr, 1)) ) return(result);
//		ptr++;
//	}
//	return(0);

	// Since we happen to have all of the bytes in one buffer, the fast way is to make one call
	// to MidiWriteBytes() for that block
	return(MidiWriteBytes(&MidiFile, ptr, MidiFile.EventSize));
}





// ****************************** OnMetaSeqNum() ********************************
// * This is called by MIDIFILE.DLL after it has called OnStartMTrk (ie, the MTrk
// * header has been written) but before OnStandardEvt gets called. In other words,
// * this gives us a chance to write the first event in an MTrk. Usually, this is
// * the Sequence Number event, which MUST be first. If we had a Sequence Number
// * event at the start of this example's internal data, we wouldn't need this,
// * since OnStandardEvt would write that out. But, this gives us an opportunity
// * to illustrate how to write such an event now. We simply format the METASEQ
// * for the DLL to write a Seq Number event followed by a Track Name event.
// *
// * If we wished to write other events at the head of the MTrk, but which aren't
// * actually events in our example track, we would need to make calls to
// * MidiWriteEvt(). We would have to do this for the sequence number since that
// * event must come first in an MTrk. Then, we would do MidiWriteEvt for all of
// * our other desired "contrived events", except for the last one. We must always
// * return at least one event in the MIDIFILE structure.
// *
// * NOTE: MIDIFILE's Time field is 0 upon entry. Do not write out any events with
// * a non-zero Time here. Do that in standardEvt().
// ******************************************************************************

long APIENTRY MfcMidiFile::OnMetaSeqNum()
{
//	long		val;
	METASEQ *	mf;

	// Fool compiler into regarding the METAFILE as a METASEQ
	mf = (METASEQ *)&MidiFile;

	// Seq Number Meta-Event
	mf->Type = 0xFF;
	mf->WriteType = 0x00;

	// Arbitrarily set Seq # to 10 + TrackNum
	mf->SeqNum = 10+mf->TrackNum;

	// Set NamePtr to point to the null-terminated Track Name to write as a Meta-Event.
	// Alternately, we could set this to 0 if we don't want a Name event to be written (or if
	// we already have an explicit Name event within our track data, to be written out in
	// OnStandardEvt)
	mf->NamePtr = &names[mf->TrackNum][0];

	// This code shows how we might write additional events. Commented out

	// Write the sequence num and name events
//	if ((val = MidiWriteEvt(&MidiFile))) return(val);

	// We could write out more events here.

	// Let DLL write the last of the events to be written out at Time of 0. In this case, a text
	// event
//	MidiFile.Status = 0xFF;
//	MidiFile.Data[0] = 0x01;
//	MidiFile.EventSize = 0;
//	set_ptr((unsigned long *)&MidiFile.Data[2], "More text");

	// Success
	return(0);
}






// **************************** OnUnknownChunk() *****************************
// * This is called by MIDIFILE.DLL after all of the MTrk chunks have been
// * successfully written out. We could use this to write a proprietary chunk
// * not defined by the MIDI File spec (although a better way to store
// * proprietary info is within a MTrk with the Proprietary Meta-Event since
// * badly written MIDI file readers might barf on an undefined chunk --
// * MIDIFILE.DLL doesn't do that). Just for the sake of illustration, we'll
// * write out a chunk with the ID of "XTRA" and it will contain 10 bytes,
// * numbers 0 to 9.
// ***************************************************************************


long APIENTRY MfcMidiFile::OnUnknownChunk()
{
	long			val;
	unsigned short	i;
	unsigned char	chr;

	// Set the ID. Note that the ULONG contains the 4 ascii chars of 'X', 'T', 'R', and 'A',
	// except that the order has been reversed to 'A', 'R', 'T', and 'X'. This reversal is due to
	// the backwards byte order of the Intel CPU when storing longs. (Alternately, we could store it
	// in Intel format and use MidiFlipLong() to reverse to Motorola big endian
	MidiFile.ID = 0x41525458;

	// Set the ChunkSize to 0 initially. We write an empty header, then after we write out all
	// of the bytes, MidiCloseChunk will automatically set the proper ChunkSize.
	MidiFile.ChunkSize = 0;

	// Write the header
	if ((val = MidiWriteHeader(&MidiFile))) return(val);

	// Write the data bytes
	for (i=0; i<10; i++)
	{
		chr = i+'0';
		if ((val = MidiWriteBytes(&MidiFile, &chr, 1))) return(val);
	}

	// Close the chunk
	MidiCloseChunk(&MidiFile);

	// Of course, we could write out more chunks here


	// Success
	return(0);
}





// **************************** MfcMidiFile() **************************
// MfcMidiFile constructor. Calls the CMidiFile constructor, and then
// initializes the MIDICALL fields with our callback function addresses.

MfcMidiFile::MfcMidiFile() : CMidiFile()
{

#ifdef BUFFERED_IO
	// I'll do my own buffered Open, Read, Seek, and Close
	ReadWriteMidi = (long (__stdcall CMidiFile::*)(unsigned char *,unsigned long))OnReadWriteMidi;
	OpenMidi = (long (__stdcall CMidiFile::*)(void))OnOpenMidi;
	SeekMidi = (long (__stdcall CMidiFile::*)(long))OnSeekMidi;
	CloseMidi = (long (__stdcall CMidiFile::*)(void))OnCloseMidi;
#endif
#ifdef RAMFILE_IO
	// I'll do my own Open, Read, Seek, and Close reading from RAM
	ReadWriteMidi = (long (__stdcall CMidiFile::*)(unsigned char *,unsigned long))OnReadWriteMidi;
	OpenMidi = (long (__stdcall CMidiFile::*)(void))OnOpenMidi;
	SeekMidi = (long (__stdcall CMidiFile::*)(long))OnSeekMidi;
	CloseMidi = (long (__stdcall CMidiFile::*)(void))OnCloseMidi;
#endif
	// Store Callback function ptrs for MIDIFILE.DLL
	StartMTrk = (long (__stdcall CMidiFile::*)(void))OnStartMTrk;
	UnknownChunk = (long (__stdcall CMidiFile::*)(void))OnUnknownChunk;
//	MetaText = (long (__stdcall CMidiFile::*)(void))OnMetaText;  // Not needed since we always supply a data buffer to the DLL
	SysexEvt = (long (__stdcall CMidiFile::*)(void))OnSysexEvt;
	StandardEvt = (long (__stdcall CMidiFile::*)(void))OnStandardEvt;
	MetaSeqNum = (long (__stdcall CMidiFile::*)(void))OnMetaSeqNum;
}





// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// What follows are just some extra callbacks you can add for some optional
// features; to add buffered file I/O, or to create a MIDI file in RAM.

#ifdef BUFFERED_IO

// **************************** OnOpenMidi() **************************
// Demonstrates adding a callback whereby we open the MIDI file using
// standard C file I/O, fopen(), for buffered I/O, rather than letting
// the DLL open the MIDI file. We also add OnReadWriteMidi, OnSeekMidi,
// and OnCloseMidi callbacks to manage our own file I/O instead of
// relying upon the DLL to write and seek bytes. Note that we could have
// opened the file back in SaveMidi before we called MidiWriteFile(), but
// we'd still need an OpenMidi callback which did nothing more than return
// 0.

long APIENTRY MfcMidiFile::OnOpenMidi()
{
	struct _stat buffer;

	// Retrieve the filename ptr which we stored in the MIDIFILE handle field, and open the file
	if(!(MidiFile.Handle  = (FILE *)fopen((const char *)MidiFile.Handle, "wb" )))
	{
		// If we wanted to force MidiWriteFile() to return some error code (to SaveMidi) that we could
		// distinguish from an error that the DLL returns, we could return some negative number here.
		// The DLL does not return any negative error numbers via MidiWriteFile(), but our callback can
		// do that. Nevertheless, I'll just return an appropriate error code that the DLL would return
		// if it couldn't open a file, and thus, we'll simply display the DLL's error message. Note that
		// this will abort the write operation on return.
		return(MIDIERRFILE);
	}

	// We must also set MidiFile.FileSize
	MidiFile.FileSize = 0;

	// Success
	return(0);
}






// ************************* OnReadWriteMidi() *************************
// Demonstrates adding a callback whereby we write bytes to the MIDI
// file using standard C file I/O, fwrite(), for buffered I/O, rather than
// letting the DLL write to the MIDI file.

long APIENTRY MfcMidiFile::OnReadWriteMidi(unsigned char * Buffer, unsigned long Count)
{
	if (fwrite((void *)Buffer, sizeof(char), Count, (FILE *)MidiFile.Handle) != Count)
	{
		// If we wanted to force MidiWriteFile() to return some error code (to SaveMidi) that we could
		// distinguish from an error that the DLL returns, we could return some negative number here (other
		// than -1, which this particular callback is not allowed to return). The DLL does not return any
		// negative error numbers via MidiWriteFile(), but our callback can do that. Nevertheless, I'll just
		// return an appropriate error code that the DLL would return if it couldn't write to a file, and
		// thus, we'll simply display the DLL's error message. Note that this will abort the write operation
		// on return.
		return(MIDIERRWRITE);
	}

	// Success
	return(0);
}





// **************************** OnSeekMidi() ****************************
// Demonstrates adding a callback whereby we seek within the MIDI
// file using standard C file I/O, fseek(), for buffered I/O, rather
// than letting the DLL seek within the MIDI file.

long APIENTRY MfcMidiFile::OnSeekMidi(long Offset)
{
	if (fseek((FILE *)MidiFile.Handle, Offset, SEEK_CUR))
	{
		// If we wanted to force MidiWriteFile() to return some error code (to SaveMidi) that we could
		// distinguish from an error that the DLL returns, we could return some negative number here.
		// The DLL does not return any negative error numbers via MidiWriteFile(), but our callback can
		// do that. Nevertheless, I'll just return an appropriate error code that the DLL would return
		// if it couldn't seek, and thus, we'll simply display the DLL's error message. Note that
		// this will abort the write operation on return.
		return(MIDIERRSEEK);
	}

	// Success
	return(0);
}





// **************************** OnCloseMidi() **************************
// Demonstrates adding a callback whereby we close the MIDI file using
// standard C file I/O, fclose(), for buffered I/O, rather than letting
// the DLL close the MIDI file. Note that we don't have to close the
// file here if we want it to be left open when MidiWriteFile returns
// back in SaveMidi. But we'd still need a CloseMidi callback which did
// nothing.

long APIENTRY MfcMidiFile::OnCloseMidi()
{
	// Close the file. Note that this never gets called if my OnOpenMidi callback returned an error.
	// But, if any other subsequent errors occur, this gets called before the write operation
	// aborts. So, I can always be assured that the file handle gets closed, even if an error
	// occurs
	fclose((FILE *)MidiFile.Handle);

	// Success. (Actually, we don't need a return for this callback, but it makes it easier to setup the
	// CALLBACK structure)
	return(0);
}
#endif





#ifdef RAMFILE_IO

unsigned char * RAM_Pointer;

// **************************** OnOpenMidi() **************************
// Demonstrates adding a callback whereby we setup to create a MIDI file
// that is in RAM, rather than letting the DLL open a MIDI file on disk.
// We also add OnReadWriteMidi, OnSeekMidi, and OnCloseMidi callbacks to
// manage our creation of the file in RAM instead of relying upon the DLL
// to write and seek to disk.

long APIENTRY MfcMidiFile::OnOpenMidi()
{
	// Allocate a block of uncommitted memory (4096 will be enough for this app's purposes) which we can
	// grow as the DLL hands us bytes to store in this RAM. In fact, let's store the pointer to this
	// allocated mem in the MIDIFILE's Handle field. We'll also initialize a global pointer to the start
	// of our MIDI file in RAM which can be used by our ReadWriteMidi and SeekMidi callbacks. Note that
	// we could have done all this back in SaveMidi before calling MidiWriteFile, but
	// we'd still need an OpenMidi callback which did nothing but return 0.

	if ((MidiFile.Handle = (HANDLE *)VirtualAlloc(0, 4096, MEM_RESERVE, PAGE_READWRITE)))
	{
		RAM_Pointer = (unsigned char *)MidiFile.Handle;

		// We must also set MidiFile.FileSize
		MidiFile.FileSize = 0;

		// Success
		return(0);
	}

	// Let's return our own defined error number to indicate "Can't allocate memory". MidiGetErr() won't have
	// an error message to display for this, so SaveMidi will have to construct its own message
	return(-3);
}






// ************************* OnReadWriteMidi() *************************
// Demonstrates adding a callback whereby we setup to create a MIDI file
// that is in RAM, rather than letting the DLL write to a MIDI file on disk.

long APIENTRY MfcMidiFile::OnReadWriteMidi(unsigned char * Buffer, unsigned long Count)
{
	// See if our global pointer is currently set so that writing Count bytes
	// would put us past the end of our allocated memory. If so, we need to
	// "grow" that memory before writing to it
	if ((RAM_Pointer + Count) >= ((unsigned char)MidiFile.Handle + MidiFile.FileSize))
	{
		// Grow our RAM block by however many bytes the DLL wants us to save
		if (!VirtualAlloc(((void *)RAM_Pointer, Count, MEM_COMMIT, PAGE_READWRITE))
		{
			// Let's return our own defined error number to indicate "Can't allocate memory". MidiGetErr() won't have
			// an error message to display for this, so SaveMidi will have to construct its own message
			return(-3);
		}
	}

	// Copy those bytes from the DLL supplied buffer to our RAM buffer
	memcpy(RAM_Pointer, Buffer, Count);
	
	// Update our RAM pointer
	RAM_Pointer += Count;
	
	// Success
	return(0);
}





// **************************** OnSeekMidi() ****************************
// Demonstrates adding a callback whereby we setup to create a MIDI file
// that is in RAM, rather than letting the DLL write to a MIDI file on disk.

long APIENTRY MfcMidiFile::OnSeekMidi(long Offset)
{
	// Adjust our RAM pointer by the Offset
	RAM_Pointer += Offset;
	
	// Success
	return(0);
}





// **************************** OnCloseMidi() **************************
// Demonstrates adding a callback whereby we setup to create a MIDI file
// that is in RAM, rather than letting the DLL write to a MIDI file on disk.

long APIENTRY MfcMidiFile::OnCloseMidi()
{
	// Normally, we'd do nothing here (but this callback must still be supplied) and let MidiWriteBytes
	// return to SaveMidi. SaveMidi could then retrieve the mem block address from MIDIFILE's Handle
	// field and the size of the data from MIDIFILE's FileSize field. SaveMidi would be responsible for
	// freeing the memory (including if an error occurred. For best results, it would be best to have
	// SaveMidi allocate the uncommitted mem prior to calling MidiWriteFile, rather than having our
	// OpenMidi callback do that).

	// Here let's just free the memory
	if (MidiFile.Handle) VirtualFree((void *)MidiFile.Handle, MidiFile.FileSize, MEM_DECOMMIT);

	// Success. (Actually, we don't need a return for this callback, but it makes it easier to setup the
	// CALLBACK structure)
	return(0);
}
#endif
