/* DelPgms.c
 *
 * An example of using GENMIDI.DLL.
 *
 * This example shows how to call the DLL functions MidiDelInstrument(),
 * MidiDelBank(), MidiDelPgm(). and MidiDelNote() to delete
 * an Instrument, Bank, and Program or Note in the DLL's loaded
 * MIDIPGMS.DAT file.
 *
 * It also demonstrates how to traverse the entire DLL database, listing
 * each instrument with its banks, programs, and note names.
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "..\genmidi.h"

long	lines = 0;
long	count = 1;

unsigned char DefOct = 3;


/************************** check_pause() ****************************
 * Check if we have filled the screen. If so, pause output waiting for
 * the user to press a key to continue.
 *********************************************************************/

void check_pause(long more)
{
	/* User wants the pause feature (ie, lines is not 0), and we filled the screen? */
	count += more;
	if (lines && (count >= lines))
	{
		count = 1;
		_cputs("--- Press ENTER to continue ---");
		getchar();
		_cputs("\r\n");
	}
}



/***************************** main() ********************************/

void main(int argc, char *argv[])
{
	unsigned char *	deleteArray[4];
	unsigned char *	ptr;
	unsigned char *	noteName;
	unsigned long	loop;
	unsigned char	i, insNum, noteNum;
	unsigned short	bankNum;

	/* Assume we won't delete any Instrument, Bank, Program, or Note */
	deleteArray[0] = deleteArray[1] = deleteArray[2] = deleteArray[3] = 0;
	i = 0;

	/* Parse command line args */
	for (loop=1; loop < (unsigned long)argc; loop++)
	{
		/* An option? */
		ptr = argv[loop];
		if (*ptr == '/')
		{
			ptr++;
			switch ((*ptr) & 0x5F)
			{
				/* Screen lines */
				case 'L':
					lines = atol(ptr+1);
			}
		}

		/* Instrument, Bank, Program, or Note name to delete */
		else if (i < 3)
		{
			deleteArray[i] = ptr;
			i++;
		}
	}

	/* Lock the database before changes. Indicate that we wish to change it */
	if (!MidiLock(MIDILOCK_MODIFY, MIDILOCK_WAITFOREVER))
	{
		/* An Instrument name supplied? */
		if (deleteArray[0])
		{
			/* Check that this instrument name really is in the database, and get its number. Also let
			   DLL set MidiNames.Instrument */
			insNum = (unsigned char)MidiGetInstrumentNum(deleteArray[0]);

			/* A Bank name supplied? */
			if (deleteArray[1])
			{
				/* Check that this bank name really is in the database, and get its number. Also, let
				   DLL set MidiNames.Instrument and MidiNames.Bank */
				bankNum = (unsigned short)MidiGetBankNum(deleteArray[1], insNum);

				/* Check that this program name really is in the database, and let DLL set
				   MidiNames.Instrument, MidiNames.Bank, and MidiNames.Pgm */
				if (deleteArray[2])
				{
					MidiGetPgmNum(deleteArray[2], bankNum, insNum);
				}
			}
		}

		switch(i)
		{
			/* Only Instrument specified. Delete that */
			case 1:
				/* Note: MidiNames.Instrument set from above call to MidiGetInstrumentNum() */
				MidiDelInstrument(MidiNames.Instrument);
				break;
		
			/* Instrument and bank specified. Delete that bank */
			case 2:
				/* Note: MidiNames.Instrument and MidiNames.Bank set from above call to MidiGetBankNum() */
				MidiDelBank(MidiNames.Instrument, MidiNames.Bank);
				break;
		
			/* Instrument, bank, and program specified. Delete that program */
			case 3:
				/* Note: MidiNames.Instrument, MidiNames.Bank and MidiNames.Pgm set from above call
				   to MidiGetPgmNum() */
				MidiDelPgm(MidiNames.Instrument, MidiNames.Bank, MidiNames.Pgm);
				break;

			/* Instrument, bank, program, and note specified. Delete that note */
			case 4:
				/* Check that this note name exists, and let DLL set MidiNames.Note */
				if (MidiNames.Pgm && (noteNum = MidiGetNoteNum(deleteArray[3], MidiNames.Pgm, &DefOct)) < 128)
				{
					MidiDelNote(MidiNames.Instrument, MidiNames.Bank, MidiNames.Pgm, MidiNames.Note);
				}
		}

		/* Unlock the database after deletions */
		MidiUnlock();
	}

	/* ******************* Now print out the GENMIDI.DLL database **************** */

	/* Indicate that we're accessing the database, but don't lock out other programs */
	if (!MidiLock(MIDILOCK_ACCESS, MIDILOCK_WAITFOREVER))
	{
		/* Start with first instrument */
		deleteArray[0] = 0;

		/* Get next Instrument name */
		while (MidiNextInstrument(&deleteArray[0]))
		{
			/* Print the instrument name */
			_cputs("===============================================================\r\n");
			_cputs(deleteArray[0]);
			_cputs("\r\n");

			check_pause(3);

			/* Start with first bank */
			deleteArray[1] = 0;

			/* Get next Bank name */
			while (MidiNextBank(&deleteArray[1], deleteArray[0]))
			{
				/* Print the Bank name */
				_cputs("\t-------------------------------------------------------\r\n\t");
				_cputs(deleteArray[1]);
				_cputs("\r\n\t-------------------------------------------------------\r\n");

				check_pause(3);

				deleteArray[2] = 0;

				/* Get the next patch name */
				while (MidiNextPgm(&ptr, deleteArray[1], &deleteArray[2]) != 0xFF)
				{
					/* Print the patch name */
					_cputs("\t\t");
					_cputs(ptr);
					_cputs("\r\n");

					check_pause(1);

					deleteArray[3] = 0;
					/* Get the next note name */
					while ((noteName = MidiNextNote(&deleteArray[3], ptr)))
					{
						/* Print the note name */
						_cputs("\t\t\t");
						_cputs(noteName);
						_cputs("\r\n");

						check_pause(1);
					}
				}
			}
		}

		/* Indicate that we're done accessing the database */
		MidiUnlock();
	}

	exit(0);
}
